﻿'The DatabaseForm class is responsible for the main application interface that is opened when
'the application is launched
'
'It creates a single EViewsDatabaseManager class during startup which is held until the application
'is shut down. This database manager class is used to create a new EViewsDatabase class each time
'an EViews workfile or database is opened.
'

Imports System.Runtime.InteropServices ' for ComException

Public Class DatabaseForm
    'option setting for whether application should use dd/mm/yyyy format for weekly/daily/intraday data
    Public Const bDayBeforeMonth As Boolean = False

    'handle to the EViews database manager class
    Private dbMgr As EViewsEdx.IDatabaseManager = Nothing

    'handle to the currently open database (if any)
    Private db As EViewsEdx.IDatabase = Nothing

    'read-write mode of the currently open database
    Private rwMode As EViewsEdx.ReadWriteMode = EViewsEdx.ReadWriteMode.FileReadWrite

    'Process errors generated by the EViews Database interface
    Private Sub ProcessComError(errorInfo As COMException)
        MsgBox(errorInfo.Message)
    End Sub

    'Initialization code when the form is first loaded
    Private Sub Form1_Load(sender As Object, e As EventArgs) Handles Me.Load
        'construct the database manager class
        Try
            dbMgr = New EViewsEdx.EViewsDatabaseManager
        Catch errorInfo As COMException
            ProcessComError(errorInfo)
            Exit Try
        End Try

        'Enable/disable controls
        AdjustOpenClose()

    End Sub

    'Enable/disable controls based on whether a database is open and
    'whether it is read-write or read-only
    Private Sub AdjustOpenClose()
        If (db Is Nothing) Then
            'no database is currently open
            If (FileNameTextBox.Text.Length > 0) Then
                OpenDb.Enabled = True
            Else
                OpenDb.Enabled = False
            End If
            CloseDb.Enabled = False
            Search.Enabled = False
            FileNameTextBox.Enabled = True
            Browse.Enabled = True

            SearchTextBox.Enabled = False
            Search.Enabled = False

            NameList.Enabled = False

            OpenObject.Enabled = False

            NewObject.Enabled = False
            RenameObject.Enabled = False
            CopyObject.Enabled = False
            DeleteObject.Enabled = False

        Else
            'a database is currently open
            OpenDb.Enabled = False
            CloseDb.Enabled = True
            Search.Enabled = True
            FileNameTextBox.Enabled = False
            Browse.Enabled = False

            SearchTextBox.Enabled = True
            Search.Enabled = True

            NameList.Enabled = True

            AdjustObjectActionButtons()

        End If
    End Sub

    'adjust the object action buttons
    Private Sub AdjustObjectActionButtons()
        If (NameList.SelectedItems.Count > 0) Then
            OpenObject.Enabled = True
        Else
            OpenObject.Enabled = False
        End If

        If (rwMode = EViewsEdx.ReadWriteMode.FileReadWrite) Then
            NewObject.Enabled = True
        Else
            NewObject.Enabled = False
        End If


        If (NameList.SelectedItems.Count > 0 And rwMode = EViewsEdx.ReadWriteMode.FileReadWrite) Then
            NewObject.Enabled = True
            CopyObject.Enabled = True
            RenameObject.Enabled = True
            DeleteObject.Enabled = True
        Else
            CopyObject.Enabled = False
            RenameObject.Enabled = False
            DeleteObject.Enabled = False
        End If
    End Sub

    'discard resources
    Private Sub Form1_FormClosing(sender As Object, e As FormClosingEventArgs) Handles Me.FormClosing
        'close the database
        Try
            If (Not db Is Nothing) Then
                db.Close()
                db = Nothing
            End If
        Catch errorInfo As COMException
            ProcessComError(errorInfo)
            Exit Try
        End Try

        'release the database manager
        Try
            If (Not dbMgr Is Nothing) Then
                dbMgr.Close()
                dbMgr = Nothing
            End If
        Catch errorInfo As COMException
            ProcessComError(errorInfo)
            Exit Try
        End Try

    End Sub

    'browse for a file name, then open an existing database or create a new database
    Private Sub Browse_Click(sender As Object, e As EventArgs) Handles Browse.Click
        With OpenFileDialog1
            .Filter = "EViews Database (*.edb)|*.edb|EViews Workfile (*.wf1)|*.wf1|All files (*.*)|*.*"
            .DefaultExt = ".edb"
            .FileName = FileNameTextBox.Text
        End With

        If (OpenFileDialog1.ShowDialog() = Windows.Forms.DialogResult.OK) Then
            Me.FileNameTextBox.Text = OpenFileDialog1.FileName
            If (OpenFileDialog1.ReadOnlyChecked) Then
                Me.rwMode = EViewsEdx.ReadWriteMode.FileReadOnly
            Else
                Me.rwMode = EViewsEdx.ReadWriteMode.FileReadWrite
            End If
            OpenDatabase()
        End If

    End Sub

    'user clicked on 'open' button to open the current database
    Private Sub Open_Click(sender As Object, e As EventArgs) Handles OpenDb.Click
        OpenDatabase()

    End Sub

    Private Sub OpenDatabase()
        Dim errorCode As EViewsEdx.ErrorCode
        db = Nothing

        'try to open database in requested mode
        Try
            errorCode = 0
            db = dbMgr.OpenDb(FileNameTextBox.Text, EViewsEdx.OpenCreateMode.FileOpen, rwMode, "", "", "")
        Catch errorInfo As COMException
            errorCode = errorInfo.ErrorCode
            If (errorCode = EViewsEdx.ErrorCode.FILE_FILENAME_INVALID) Then
                'file doesn't exist. prompt and create?
            ElseIf (rwMode = EViewsEdx.ReadWriteMode.FileReadWrite And errorCode = EViewsEdx.ErrorCode.FILE_ACCESS_DENIED) Then
                'couldn't get write handle. try for read only?
            Else
                ProcessComError(errorInfo)
            End If
            Exit Try
        End Try

        If (errorCode = EViewsEdx.ErrorCode.FILE_FILENAME_INVALID) Then
            'database doesn't exist. create it?
            If (MsgBox("Database does not exist. Create it now?", MsgBoxStyle.YesNo) = MsgBoxResult.Yes) Then
                Try
                    rwMode = EViewsEdx.ReadWriteMode.FileReadWrite
                    errorCode = 0
                    db = dbMgr.OpenDb(FileNameTextBox.Text, EViewsEdx.OpenCreateMode.FileCreate, rwMode, "", "", "")
                Catch errorInfo As COMException
                    ProcessComError(errorInfo)
                    Exit Try
                End Try
            End If
        ElseIf (rwMode = EViewsEdx.ReadWriteMode.FileReadWrite And errorCode = EViewsEdx.ErrorCode.FILE_ACCESS_DENIED) Then
            'unable to open in read-write mode. try read only?
            If (MsgBox("Unable to open database in read-write mode. Switch to read-only mode?", MsgBoxStyle.YesNo) = MsgBoxResult.Yes) Then
                rwMode = EViewsEdx.ReadWriteMode.FileReadOnly
                Try
                    errorCode = 0
                    db = dbMgr.OpenDb(FileNameTextBox.Text, EViewsEdx.OpenCreateMode.FileOpen, rwMode, "", "", "")
                Catch errorInfo As COMException
                    ProcessComError(errorInfo)
                    Exit Try
                End Try
            End If
        End If

        If (errorCode = 0) Then
            AdjustOpenClose()

            'move cursor to search field
            If (SearchTextBox.Text.Length = 0) Then
                SearchTextBox.Text = "*"
            End If
            SearchTextBox.Focus()

        End If

    End Sub

    Private Sub FileNameTextBox_TextChanged(sender As Object, e As EventArgs) Handles FileNameTextBox.TextChanged
        AdjustOpenClose()
    End Sub

    'user clicked on 'Close' to close the current database
    Private Sub Close_Click(sender As Object, e As EventArgs) Handles CloseDb.Click
        CloseDatabase()
    End Sub

    Private Sub CloseDatabase()
        'close database object
        Try
            db.Close()
            db = Nothing
        Catch errorInfo As COMException
            ProcessComError(errorInfo)
            Exit Try
        End Try

        'adjust user interface
        AdjustOpenClose()
        NameList.Items.Clear()

        'move cursor to filename
        FileNameTextBox.Focus()
    End Sub

    'user clicked on the 'Search' button
    Private Sub Search_Click(sender As Object, e As EventArgs) Handles Search.Click
        SearchDb()
    End Sub

    'retrieve items that match search
    Private Sub SearchDb()
        Try
            NameList.Items.Clear()

            Cursor = Cursors.WaitCursor

            'initialize search
            db.SearchByAttributes("name matches " & SearchTextBox.Text, "name, type")

            'retrieve results
            Dim resultCount As Integer = 0
            Dim name As String = Nothing
            Dim attr As Object = Nothing
            While (db.SearchNext(name, attr))
                NameList.Items.Add(name)
                resultCount = resultCount + 1
            End While

            Cursor = Cursors.Default

            If (resultCount = 0) Then
                MsgBox("No objects found.")
            End If

        Catch errorInfo As COMException
            ProcessComError(errorInfo)
            Exit Try
        End Try

        AdjustOpenClose()

        'move cursor to filename
        FileNameTextBox.Focus()
    End Sub

    Private Sub SearchTextBox_KeyDown(sender As Object, e As KeyEventArgs) Handles SearchTextBox.KeyDown
        If (e.KeyCode = Keys.Enter) Then
            e.Handled = True
            e.SuppressKeyPress = True
            SearchDb()
        End If
    End Sub

    'user clicked on 'Delete' to remove an object from the database
    Private Sub Delete_Click(sender As Object, e As EventArgs) Handles DeleteObject.Click
        If (NameList.SelectedItems.Count > 0) Then
            If (MsgBox("Delete object " & StrConv(NameList.SelectedItem.ToString(), VbStrConv.Uppercase) & "?", MsgBoxStyle.OkCancel) = MsgBoxResult.Ok) Then
                Try
                    db.DeleteObject(NameList.SelectedItem.ToString())

                    'remove from list
                    NameList.Items.Remove(NameList.SelectedItem)

                Catch errorInfo As COMException
                    ProcessComError(errorInfo)
                    Exit Try
                End Try
            End If
        End If
    End Sub

    'user clicked on 'Rename' to change the name of an object in the database
    Private Sub Rename_Click(sender As Object, e As EventArgs) Handles RenameObject.Click
        If (NameList.SelectedItems.Count > 0) Then
            Dim OldName As String = NameList.SelectedItem.ToString()
            Dim NewName As String = OldName

            Dim Status As Integer = 0
            While (Status = 0)
                'prompt for new name
                NewName = InputBox("Rename " & StrConv(OldName, VbStrConv.Uppercase) & " to:", "Rename Object", NewName)
                If (NewName.Length > 0 And StrConv(NewName, VbStrConv.Uppercase) <> StrConv(OldName, VbStrConv.Uppercase)) Then
                    'new name is different from existing name
                    Try
                        db.RenameObject(NameList.SelectedItem.ToString(), NewName)
                        Status = 1

                    Catch errorInfo As COMException
                        ProcessComError(errorInfo)
                        Exit Try
                    End Try
                Else
                    'cancel button clicked or new name is the same as the existing name
                    Status = -1
                End If
            End While

            If (Status = 1) Then
                'adjust item in name list
                Dim SelectedItemNo = NameList.Items.IndexOf(NameList.SelectedItem)
                NameList.Items(SelectedItemNo) = NewName
            End If
        End If

    End Sub

    'user clicked on 'Copy' to create a copy of an object in the database
    Private Sub Copy_Click(sender As Object, e As EventArgs) Handles CopyObject.Click
        If (NameList.SelectedItems.Count > 0) Then
            Dim OldName As String = NameList.SelectedItem.ToString()
            Dim NewName As String = OldName

            Dim Status As Integer = 0
            While (Status = 0)
                'prompt for new name
                NewName = InputBox("Copy " & StrConv(OldName, VbStrConv.Uppercase) & " as:", "Copy Object", NewName)
                If (NewName.Length > 0 And StrConv(NewName, VbStrConv.Uppercase) <> StrConv(OldName, VbStrConv.Uppercase)) Then
                    'new name is different from existing name
                    Dim errorCode As EViewsEdx.ErrorCode = 0
                    Try
                        'copy without overwrite
                        db.CopyObject(NameList.SelectedItem.ToString(), NewName, False)
                        Status = 1

                    Catch errorInfo As COMException
                        errorCode = errorInfo.ErrorCode
                        If (errorCode <> EViewsEdx.ErrorCode.RECORD_NAME_IN_USE) Then
                            ProcessComError(errorInfo)
                        End If
                        Exit Try
                    End Try

                    'if existing object found, prompt for whether to overwrite
                    If (errorCode = EViewsEdx.ErrorCode.RECORD_NAME_IN_USE) Then
                        If (MsgBox("Object " & StrConv(NewName, VbStrConv.Uppercase) & " already exists. Overwrite the existing object?", MsgBoxStyle.OkCancel) = MsgBoxResult.Ok) Then
                            Try
                                'user has confirmed overwrite, so do it
                                db.CopyObject(NameList.SelectedItem.ToString(), NewName, True)
                                Status = 2
                            Catch errorInfo As COMException
                                ProcessComError(errorInfo)
                                Exit Try
                            End Try
                        End If
                    End If
                Else
                    'cancel button clicked or new name is the same as the existing name
                    Status = -1
                End If
            End While

            If (Status = 1) Then
                'add new name to list
                NameList.Items.Add(NewName)
            End If
        End If
    End Sub

    Private Sub Show_Click(sender As Object, e As EventArgs) Handles OpenObject.Click
        ShowObject()
    End Sub

    'user clicked on 'Show' to display/edit an existing object in the database
    Private Sub ShowObject()
        If (NameList.SelectedItems.Count > 0) Then
            Dim objectName As String = NameList.SelectedItem.ToString()
            Try
                Dim attr As Object = Nothing
                Dim vals As Object = Nothing
                Dim ids As Object = Nothing
                db.ReadObject(objectName, "", attr, vals, ids)

                'create field browse dialog
                Dim showObjectForm As New ObjectForm(objectName, attr, vals, ids, False, bDayBeforeMonth)

                'show dialog
                Dim result As System.Windows.Forms.DialogResult = showObjectForm.ShowDialog()

                If (result = Windows.Forms.DialogResult.OK) Then
                    'user asked to save changes: try saving the object back into the database (overwriting the existing object)
                    db.WriteObject(objectName, showObjectForm.attr, showObjectForm.vals, showObjectForm.ids, EViewsEdx.WriteType.WriteOverwrite)
                End If

            Catch errorInfo As COMException
                ProcessComError(errorInfo)
                Exit Try
            End Try
        End If
    End Sub

    'handle double click on object name by showing object
    Private Sub NameList_MouseDoubleClick(sender As Object, e As MouseEventArgs) Handles NameList.MouseDoubleClick
        ShowObject()
    End Sub

    'user clicked on 'Add' to create a new object in the database
    Private Sub Add_Click(sender As Object, e As EventArgs) Handles NewObject.Click
        'create field browse dialog
        Dim showNewSeriesdialog As New NewSeriesDialog

        'show dialog
        If (showNewSeriesdialog.ShowDialog() = Windows.Forms.DialogResult.OK) Then
            'build up attribute list for new object
            Dim name As String = showNewSeriesdialog.objectName
            Dim attr As Object = New Object(5, 1) {}
            attr(0, 0) = "type"
            attr(0, 1) = showNewSeriesdialog.objectType
            attr(1, 0) = "freq"
            attr(1, 1) = showNewSeriesdialog.objectFrequency
            attr(2, 0) = "start"
            attr(2, 1) = showNewSeriesdialog.ObjectStartDate
            Dim vals As Object = Nothing
            Dim ids As Object = Nothing

            'create the object dialog so user can edit the object
            Dim showObjectForm As New ObjectForm(name, attr, vals, ids, True, bDayBeforeMonth)

            'show object dialog
            If (showObjectForm.ShowDialog() = Windows.Forms.DialogResult.OK) Then
                'user asked to save changes
                attr = showObjectForm.attr
                vals = showObjectForm.vals
                ids = showObjectForm.ids

                Dim errorCode As EViewsEdx.ErrorCode = 0
                Try
                    'try saving object back into database (error if there is an existing object)
                    db.WriteObject(name, attr, vals, ids, EViewsEdx.WriteType.WriteProtect)

                    'add name of new object to object list
                    If (Not NameList.Items.Contains(name)) Then
                        NameList.Items().Add(name)
                    End If

                Catch errorInfo As COMException
                    errorCode = errorInfo.ErrorCode
                    ProcessComError(errorInfo)
                    Exit Try
                End Try
            End If
        End If

    End Sub

    Private Sub NameList_SelectedIndexChanged(sender As Object, e As EventArgs) Handles NameList.SelectedIndexChanged
        AdjustObjectActionButtons()
    End Sub
End Class



