﻿'class used for display and editing of the attributes and observations of a single object
'
'note that this class works with a copy of the object held in memory.
'it does not use the EViewsDatabaseManager or EViewsDatabase classes.
'

Public Class ObjectForm
    'info on object being views/edited
    Private objectName As String
    Public attr As Object
    Public vals As Object
    Public ids As Object

    Public bEditOnOpen As Object    'switch into edit when dialog is open
    Public bIsDirty As Object    'true if series has been edited

    Private Frequency As EViewsEdx.IFrequency           'frequency of current object
    Private isIntegerDated As Boolean                   'is the series undated/integer dated (not calendar dated)?
    Private TimeStampFrequency As EViewsEdx.Frequency   'helper frequency for formatting timestamp info

    'data tables used for data grid controls
    Dim AttributeTable As DataTable
    Dim ObservationTable As DataTable

    'fetch an attribute value from the attributes array held in EViews EDX format
    Private Function GetAttribute(ByRef attr As Object, ByVal fieldName As String) As Object
        Dim rows As Integer = attr.GetLength(0)
        For i = 0 To rows - 1
            If (attr(i, 0) = fieldName) Then
                Return attr(i, 1)
            End If
        Next
        Return Nothing

    End Function

    Public Sub New(ByRef _objectName As String, ByRef _attr As Object, ByRef _vals As Object, ByRef _ids As Object, ByVal _bEditOnOpen As Boolean, ByVal bDayBeforeMonth As Boolean)
        ' This call is required by the designer.
        InitializeComponent()

        'save passed-in object data
        objectName = _objectName
        attr = _attr
        vals = _vals
        ids = _ids
        bEditOnOpen = _bEditOnOpen

        'use object name for Window title
        Text = StrConv(_objectName, VbStrConv.Uppercase)

        'generate a frequency class representing this object's frequency
        Dim freqString = GetAttribute(attr, "freq")
        Dim startString = GetAttribute(attr, "start")
        Frequency = New EViewsEdx.Frequency
        Frequency.Set(freqString & " " & startString, bDayBeforeMonth)
        isIntegerDated = (Not Frequency.IsValid()) Or (Frequency.ToString(False) = "U")

        'create an undated helper frequency - used for formatting timestamps
        TimeStampFrequency = New EViewsEdx.Frequency
        TimeStampFrequency.Set("U")

        'mark as unedited
        bIsDirty = False

    End Sub

    'Push data from member variables into forms
    Public Sub Populate()

        'populate attribute table
        Dim nattr As Integer = attr.GetLength(0)

        AttributeTable = New DataTable
        AttributeTable.Columns.Add("Attribute")
        AttributeTable.Columns.Add("Value")

        For i = 0 To nattr - 1

            If (Not attr(i, 0) Is Nothing) Then
                'add a row for this observation containing the attribute name and value
                Dim dr As DataRow = AttributeTable.NewRow

                Dim attrName As String = StrConv(attr(i, 0), VbStrConv.ProperCase)
                dr.Item(0) = attrName

                If (attrName = "Start" Or attrName = "End") Then
                    'format first and last attributes with standard frequency labels eg. 1980Q1, 1980Q2
                    If (Not isIntegerDated) Then
                        'calendar dates
                        Dim dateval As Date
                        If (TypeOf (attr(i, 1)) Is Date) Then
                            dateval = attr(i, 1)
                        Else
                            dateval = Frequency.StringtoD(attr(i, 1))
                        End If
                        dr.Item(1) = Frequency.DtoString(dateval)
                    Else
                        'undated / integer identifiers
                        dr.Item(1) = attr(i, 1)
                    End If
                ElseIf (attrName = "Last_Update" Or attrName = "Last_Write") Then
                    'format timestamp attributes in local (.NET) format
                    dr.Item(1) = CDate(attr(i, 1))
                Else
                    'keep all other attributes as they are
                    dr.Item(1) = attr(i, 1)
                End If

                AttributeTable.Rows.Add(dr)
            End If

        Next

        'bind the data table to the control
        AttributeView.DataSource = AttributeTable

        'make attribtue value column size to fill
        AttributeView.Columns(1).AutoSizeMode = DataGridViewAutoSizeColumnMode.Fill

        'populate data table with observation dates and values
        Dim nobs As Integer = 0
        If (Not vals Is Nothing) Then nobs = vals.GetLength(0)

        ObservationTable = New DataTable
        Dim dateCol = ObservationTable.Columns.Add("Date")
        dateCol.ReadOnly = True
        ObservationTable.Columns.Add("Value")

        For i = 0 To nobs - 1
            'add a row for this observation containing date and value
            Dim dr As DataRow = ObservationTable.NewRow
            If (Not isIntegerDated) Then
                'calendar dates
                dr.Item(0) = Frequency.DtoString(ids(i))
            Else
                'undated / integer identifiers
                dr.Item(0) = ids(i)
            End If
            dr.Item(1) = vals(i)
            ObservationTable.Rows.Add(dr)
        Next

        'bind the data table to the control
        ObservationView.DataSource = ObservationTable

        'disable sorting (too confusing when editing)
        ObservationView.Columns(0).SortMode = DataGridViewColumnSortMode.NotSortable
        ObservationView.Columns(1).SortMode = DataGridViewColumnSortMode.NotSortable
        ObservationView.Columns(1).DefaultCellStyle.Alignment = DataGridViewContentAlignment.MiddleRight

        'add tool tip to each observation date containing start and end timestamps
        If (Not isIntegerDated) Then
            For i = 0 To nobs - 1
                Dim observationNumber = Frequency.DtoO(ids(i))
                Dim startDate As String = TimeStampFrequency.DtoString(Frequency.OtoD(observationNumber, False))
                Dim endDate As String = TimeStampFrequency.DtoString(Frequency.OtoD(observationNumber, True))
                ObservationView.Rows(i).Cells(0).ToolTipText = startDate & " to " & endDate
            Next
        End If

    End Sub

    Private Sub ObjectForm_Load(sender As Object, e As EventArgs) Handles Me.Load
        'populate data tables
        Populate()

        AttributeView.CurrentCell = Nothing
        ObservationView.CurrentCell = Nothing

    End Sub

    Private Sub ObjectForm_Activated(sender As Object, e As EventArgs) Handles Me.Activated
        If (bEditOnOpen) Then
            ObservationView.Focus()
            ObservationView.CurrentCell = ObservationView.Rows(0).Cells(1)
            ObservationView.BeginEdit(False)
            bEditOnOpen = False
        End If
    End Sub


    Private Sub ObservationView_CellDoubleClick(sender As Object, e As DataGridViewCellEventArgs) Handles ObservationView.CellDoubleClick
        ObservationView.BeginEdit(True)
    End Sub

    Private Sub ObservationView_CellValueChanged(sender As Object, e As DataGridViewCellEventArgs) Handles ObservationView.CellValueChanged
        bIsDirty = True
    End Sub

    'handle delete of an observation
    Private Sub ObservationView_KeyDown(sender As Object, e As KeyEventArgs) Handles ObservationView.KeyDown
        If (e.KeyCode = Keys.Delete) Then
            Dim selectedCol As Integer = ObservationView.SelectedCells.Item(0).ColumnIndex
            Dim selectedRow As Integer = ObservationView.SelectedCells.Item(0).RowIndex
            If (selectedCol = 0) Then
                'delete observation
                ObservationView.Rows.RemoveAt(selectedRow)
            Else
                'clear value
                ObservationView.SelectedCells.Item(0).Value = Nothing
                ObservationView.BeginEdit(True)
            End If

            bIsDirty = True
        End If
    End Sub

    Private Sub AttributeView_CellDoubleClick(sender As Object, e As DataGridViewCellEventArgs) Handles AttributeView.CellDoubleClick
        AttributeView.BeginEdit(True)
    End Sub

    Private Sub AttributeView_CellValueChanged(sender As Object, e As DataGridViewCellEventArgs) Handles AttributeView.CellValueChanged
        bIsDirty = True
    End Sub

    'handle delete of an attribute
    Private Sub AttributeView_KeyDown(sender As Object, e As KeyEventArgs) Handles AttributeView.KeyDown
        If (e.KeyCode = Keys.Delete) Then
            Dim selectedCol As Integer = AttributeView.SelectedCells.Item(0).ColumnIndex
            Dim selectedRow As Integer = AttributeView.SelectedCells.Item(0).RowIndex
            If (selectedCol = 0) Then
                'delete attribute
                AttributeView.Rows.RemoveAt(selectedRow)
            Else
                'clear value
                AttributeView.SelectedCells.Item(0).Value = ""
                AttributeView.BeginEdit(True)
            End If

            bIsDirty = True
        End If
    End Sub

    'if focus enters an empty row, add a date identifier for the new observation
    Private Sub DataGridView1_RowEnter(sender As Object, e As DataGridViewCellEventArgs) Handles ObservationView.RowEnter
        Dim newRow As Integer = e.RowIndex
        If (ObservationView.Rows(newRow).Cells(0).Value Is Nothing) Then
            'put observation date in first column
            If (Not isIntegerDated) Then
                'calendar dates
                Dim newObservation As Integer
                If (newRow > 0) Then
                    'get observation number from previous row (to allow irregular dates)
                    Dim lastDate As Date = Frequency.StringtoD(ObservationView.Rows(newRow - 1).Cells(0).Value)
                    newObservation = Frequency.DtoO(lastDate) + 1
                Else
                    'first observation is always at zero
                    newObservation = 0
                End If
                ObservationView.Rows(newRow).Cells(0).Value = Frequency.DtoString(Frequency.OtoD(newObservation))

                'add tool tip containing start and end timestamps
                Dim startDate As String = TimeStampFrequency.DtoString(Frequency.OtoD(newObservation, False))
                Dim endDate As String = TimeStampFrequency.DtoString(Frequency.OtoD(newObservation, True))
                ObservationView.Rows(newRow).Cells(0).ToolTipText = startDate & " to " & endDate
            Else
                'undated / integer identifiers
                Dim newIdentifier As Integer
                If (newRow > 0) Then
                    newIdentifier = ObservationView.Rows(newRow - 1).Cells(0).Value + 1
                Else
                    newIdentifier = 0
                End If
                ObservationView.Rows(newRow).Cells(0).Value = newIdentifier
            End If
        End If

    End Sub

    Private Sub ObjectForm_FormClosing(sender As Object, e As FormClosingEventArgs) Handles Me.FormClosing
        'save any outstanding edits
        AttributeView.CommitEdit(DataGridViewDataErrorContexts.Commit)
        ObservationView.CommitEdit(DataGridViewDataErrorContexts.Commit)

        'prompt for whether to save edits
        DialogResult = Windows.Forms.DialogResult.Cancel
        If (bIsDirty) Then
            Dim result = MsgBox("Series has been edited. Save changes to database?", MsgBoxStyle.YesNoCancel)
            If (result = MsgBoxResult.Yes) Then
                'extract attribute data from data table back into attribute object
                Dim nattr As Integer = AttributeTable.Rows.Count
                ReDim attr(nattr - 1, 1)
                For i = 0 To nattr - 1
                    attr(i, 0) = AttributeTable.Rows(i).Item(0)
                    attr(i, 1) = AttributeTable.Rows(i).Item(1)
                Next

                'extract data from data table back into ids and vals
                Dim nobs As Integer = ObservationTable.Rows.Count
                ReDim ids(nobs - 1)
                ReDim vals(nobs - 1)
                For i = 0 To nobs - 1
                    If (Not isIntegerDated) Then
                        ids(i) = Frequency.StringtoD(ObservationTable.Rows(i).Item(0))
                    Else
                        ids(i) = ObservationTable.Rows(i).Item(0)
                    End If
                    vals(i) = ObservationTable.Rows(i).Item(1)
                Next

                DialogResult = Windows.Forms.DialogResult.OK
            ElseIf (result = MsgBoxResult.No) Then
                'don't save changes
                bIsDirty = False
            Else
                'user hit cancel
                e.Cancel = True
            End If
        End If
    End Sub


End Class