﻿'Implementation of an EDX Custom Browser class for EIA data. 
'
'Provides a specialized GUI for working with EIA data that exploits the EIA category information.
' 
'Note that the IDatabaseBrowserEvents interface is used to send messages to EViews. if your project uses
'.NET Framework 4.0 or later, you may need to change the ComSourceInterfaces attribute to refer to 
'EViewsEdxNet.IDatabaseBrowserEvents instead of EViewsEdx.IDatabaseBrowserEViews. The two interfaces
'are identical except that the former was declared within a .NET project while the latter was declared
'in a native code project. Versions of the .NET Framework before 4.0 seem to work reliably with the interface
'declared in native code, but later versions do not.
'

Imports System.Runtime.InteropServices

'Allow Namespace for BinaryFormatter
Imports System.IO
Imports System.Runtime.Serialization.Formatters.Binary

'implements IDatabaseBrowser class, and is a source for IDatabaseBrowserEvents.
<ClassInterface(ClassInterfaceType.None), _
ComSourceInterfaces(GetType(EViewsEdx.IDatabaseBrowserEvents)), _
ComVisible(True)> _
Public Class CategoryBrowserControl
    Implements EViewsEdx.IDatabaseBrowser

    'declaration of event used to send messages to EViews (from EViewsEdx.IDatabaseBrowserEvents)
    Public Event BrowserEvent(ByVal commandId As EViewsEdx.MessageId, ByVal commandArgs As Object, ByRef Result As Object)

    'callback interface used to communicate with the data source (EIA online or EIA bulk file database)
    Public Interface ICategorySource
        'use id=-1 to indicate root category
        Function GetCategoryInfo(categoryId As Integer, ByRef parentId As Integer, ByRef description As String, ByRef categoryPath As String, _
                                 ByRef childIds() As Integer, ByRef childDescriptions() As String, _
                                 ByRef childSeriesNames() As String, ByRef childSeriesDescriptions() As String) As Integer

        'save category that browser should start at next time it is opened
        Sub SaveStartupCategory(categoryId As Integer)

        'tell source database to prepare search results for the selected series
        Sub PrepareBrowserSelection(selectedSeriesNames() As String)

        'download a local file containing the content of this category.
        'currently only used for downloding bulk files for entire EIA data sets
        Sub DownloadCategoryContent(categoryId As Integer)

    End Interface

    'data source for category information
    Private source As ICategorySource = Nothing
    Private sourceSupportsDownloads = False

    'cached infomation on current category
    Private currentId As Integer = -1
    Private parentId As Integer = -1
    Private description As String = Nothing
    Private path As String = Nothing
    Private childCategoryIds() As Integer = Nothing
    Private childCategoryDescriptions() As String = Nothing
    Private childSeriesNames() As String = Nothing
    Private childSeriesDescriptions() As String = Nothing

    'note that statupCategory of -1 indicates to start from root category
    Public Sub New(dataSource As ICategorySource, Optional startupCategory As Integer = -1, Optional enableDownloads As Boolean = False)
        ' This call is required by the designer.
        InitializeComponent()

        ' Add any initialization after the InitializeComponent() call.
        source = dataSource

        'save startup category
        currentId = startupCategory

        sourceSupportsDownloads = enableDownloads
    End Sub

    Private Sub CategoryBrowserControl_Load(sender As Object, e As EventArgs) Handles Me.Load
        'load information for startup category into browser
        LoadCategory(currentId)

        'hide the main EViews database window
        SendMessage(EViewsEdx.MessageId.HideDbWin)

        Timer1.Enabled = True

    End Sub

    Private Sub CategoryBrowserControl_VisibleChanged(sender As Object, e As EventArgs) Handles Me.VisibleChanged
        'save current category back to data source (so browser opens on same category next time it is opened)
        source.SaveStartupCategory(currentId)
    End Sub

    'tells the browser to display a new category
    Public Sub LoadCategory(categoryId As Integer)

        'retrieve category info
        Me.Cursor = System.Windows.Forms.Cursors.WaitCursor
        Try
            currentId = source.GetCategoryInfo(categoryId, parentId, description, path, _
                                               childCategoryIds, childCategoryDescriptions, childSeriesNames, childSeriesDescriptions)
        Catch e As COMException
            'unable to fetch/read category info
            Me.Cursor = System.Windows.Forms.Cursors.Default
            MsgBox(e.Message, MsgBoxStyle.Exclamation)
            Return
        End Try

        Me.Cursor = System.Windows.Forms.Cursors.Default

        'clear out existing contents
        ChildListView.Clear()

        'add (single) dummy column to list view and switch to details view
        Dim header As Windows.Forms.ColumnHeader = New Windows.Forms.ColumnHeader()
        header.Name = "name"
        header.Text = path
        ChildListView.Columns.Add(header)
        ChildListView.View = Windows.Forms.View.Details

        'switch tool tips on
        ChildListView.ShowItemToolTips = True

        'switch off sorting while we add items
        ChildListView.Sorting = Windows.Forms.SortOrder.None

        'add special item for navigating to parent category
        Dim itemNo = 0
        If (parentId >= 0) Then
            ChildListView.Items.Add("..", "..", 0)
            ChildListView.Items(itemNo).ToolTipText = Nothing
            itemNo = itemNo + 1
        End If

        'add child categories
        For i = 0 To childCategoryIds.Count - 1
            ChildListView.Items.Add(CStr(childCategoryIds(i)), childCategoryDescriptions(i), 0)
            ChildListView.Items(itemNo).ToolTipText = Nothing
            itemNo = itemNo + 1
        Next

        'add child series
        For i = 0 To childSeriesNames.Count - 1
            ChildListView.Items.Add(childSeriesNames(i), childSeriesDescriptions(i), 1)
            ChildListView.Items(itemNo).ToolTipText = childSeriesNames(i)
            itemNo = itemNo + 1
        Next

        'sort items
        ChildListView.ListViewItemSorter = New CategoryBrowserItemComparer
        ChildListView.Sorting = Windows.Forms.SortOrder.Ascending
        ChildListView.Sort()

        'adjust column width
        ChildListView.AutoResizeColumns(2)
        If (ChildListView.Columns(0).Width < ChildListView.Width - 10) Then
            ChildListView.Columns(0).Width = ChildListView.Width - 10
        End If

        'populate menu strip with category path
        MenuStrip1.CanOverflow = True
        MenuStrip1.LayoutStyle = Windows.Forms.ToolStripLayoutStyle.Flow
        MenuStrip1.Items().Clear()
        Dim pathCategories = path.Split(">")
        For i = 0 To pathCategories.Count - 1
            Dim menuItemText As String = pathCategories(i).Trim()
            If (i < pathCategories.Count - 1) Then
                menuItemText = menuItemText & " >"
            End If
            Dim item = MenuStrip1.Items().Add(menuItemText)
            item.Overflow = Windows.Forms.ToolStripItemOverflow.AsNeeded
        Next

        'select right mouse button menu
        If (parentId >= 0 Or Not sourceSupportsDownloads) Then
            ChildListView.ContextMenuStrip = ContextMenuStrip1
        Else
            'special right mouse button 'Download' for topmost server node
            ChildListView.ContextMenuStrip = ContextMenuStrip2
        End If

    End Sub

    'single click on a list item
    Private Sub ChildListView_Click(sender As Object, e As EventArgs) Handles ChildListView.Click
        'don't process right mouse button clicks (since used for right mouse button menu)
        If (DragInitialized = 0 Or lastClickButton <> Windows.Forms.MouseButtons.Right) Then
            If (ChildListView.SelectedItems.Count > 0) Then
                Dim item As Windows.Forms.ListViewItem = ChildListView.SelectedItems.Item(0)
                Dim val = item.Name
                If (val = "..") Then
                    'user clicked on special 'parent' folder
                    LoadCategory(parentId)
                ElseIf (IsNumeric(val)) Then
                    'user clicked on category
                    LoadCategory(CInt(val))
                Else
                    'user clicked on series
                End If
            End If
        End If
    End Sub

    'double click on a list item
    Private Sub ChildListView_DoubleClick(sender As Object, e As EventArgs) Handles ChildListView.DoubleClick
        If (ChildListView.SelectedItems.Count > 0) Then
            Dim item As Windows.Forms.ListViewItem = ChildListView.SelectedItems.Item(0)
            Dim val = item.Name
            If (val = "..") Then
                'user clicked on special 'parent' folder
            ElseIf (IsNumeric(val)) Then
                'user clicked on category
            Else
                'used clicked on series - send item to EViews
                If (PrepareCurrentSelection()) Then
                    SendMessage(EViewsEdx.MessageId.PreviewSelected)
                End If
            End If
        End If
    End Sub

    'send a message to EViews
    Private Function SendMessage(ByVal commandId As EViewsEdx.MessageId, Optional ByVal commandArgs As Object = Nothing) As Object
        Dim eventResult As Object = Nothing
        RaiseEvent BrowserEvent(commandId, commandArgs, eventResult)
        Return eventResult
    End Function

    'receive events from EViews
    Public Function EViewsEvent(id As EViewsEdx.MessageId, commandArgs As Object) As Object Implements EViewsEdx.IDatabaseBrowser.EViewsEvent
        Return Nothing
    End Function

    'process a click on the menu strip (containing the category path)
    Private Sub MenuStrip1_ItemClicked(sender As Object, e As Windows.Forms.ToolStripItemClickedEventArgs) Handles MenuStrip1.ItemClicked
        'move up parent categories to clicked on category
        Dim item As System.Windows.Forms.ToolStripMenuItem = e.ClickedItem
        Dim itemNo = MenuStrip1.Items().IndexOf(item)
        Dim upsRequired = MenuStrip1.Items().Count - 1 - itemNo

        If (upsRequired > 0) Then
            For i = 1 To upsRequired - 1
                'move to parent - not particularly efficient, but should be OK since we are caching info for visited nodes
                source.GetCategoryInfo(parentId, parentId, description, path, _
                      childCategoryIds, childCategoryDescriptions, childSeriesNames, childSeriesDescriptions)
            Next
            LoadCategory(parentId)
        End If
    End Sub

    'data for drag drop support
    Private lastClickButton As Windows.Forms.MouseButtons
    Private lastClickLocation As System.Drawing.Point
    Private DragInitialized As Integer = 0      '0=up, 2=down, 1=dragging

    'keep track of which mouse button was pressed
    Private Sub ChildListView_MouseDown(ByVal sender As Object, ByVal e As System.Windows.Forms.MouseEventArgs) Handles ChildListView.MouseDown
        lastClickButton = e.Button
        lastClickLocation = e.Location
        DragInitialized = 2
    End Sub

    Private Sub ChildListView_MouseUp(ByVal sender As Object, ByVal e As System.Windows.Forms.MouseEventArgs) Handles ChildListView.MouseUp
        DragInitialized = 0
    End Sub

    'handle drag of items
    Private Sub ChildListView_MouseMove(ByVal sender As Object, ByVal e As System.Windows.Forms.MouseEventArgs) Handles ChildListView.MouseMove
        Dim dragTolerance = 2
        If ((e.Button = Windows.Forms.MouseButtons.Left Or e.Button = Windows.Forms.MouseButtons.Right) _
            And (System.Math.Abs(e.Location.X - lastClickLocation.X) >= dragTolerance Or System.Math.Abs(e.Location.Y - lastClickLocation.Y) >= dragTolerance)) Then
            'is this the first mouse move since a mouse down?
            If DragInitialized = 2 Then
                'if so, start a drag drop
                DragInitialized = 1
                If (ChildListView.SelectedItems.Count > 0) Then
                    'notify database class of user selection
                    If (PrepareCurrentSelection()) Then
                        'send message to EViews to indicate that we are beginning a drag-drop operation.
                        'EViews will return binary data to be placed inside the drag drog Data Object.
                        Dim message As EViewsEdx.MessageId
                        If (e.Button = Windows.Forms.MouseButtons.Left) Then
                            message = EViewsEdx.MessageId.DragSelected
                        Else
                            message = EViewsEdx.MessageId.DragSelectedAsLink
                        End If
                        Dim o As Object = SendMessage(message)

                        If (Not o Is Nothing) Then
                            'put data returned by EViews onto the clipboard
                            Dim b As Byte() = o
                            Dim obj As New System.Windows.Forms.DataObject
                            obj.SetData("EViewsEdxBrowserSelection", New MemoryStream(b))

                            'start drag drop
                            ChildListView.DoDragDrop(obj, Windows.Forms.DragDropEffects.Copy)

                        End If
                    End If
                End If
            End If
        End If
    End Sub

    'retrive the names of any series that are currently selected in the list view
    Private Function GetSelectedSeriesIds() As String()
        Dim seriesCount As Integer = 0
        Dim item As Windows.Forms.ListViewItem
        For Each item In ChildListView.SelectedItems()
            Dim key = item.Name
            If (Not IsNumeric(key) And key <> "..") Then
                seriesCount = seriesCount + 1
            End If
        Next

        Dim seriesNames As String() = New String(seriesCount - 1) {}
        seriesCount = 0
        For Each item In ChildListView.SelectedItems()
            Dim key = item.Name
            If (Not IsNumeric(key) And key <> "..") Then
                seriesNames(seriesCount) = key
                seriesCount = seriesCount + 1
            End If
        Next

        Return seriesNames
    End Function

    'initialize database search to return currently selected items
    Private Function PrepareCurrentSelection() As Boolean
        'get a list of the series currently selected within the browser
        Dim selectedSeriesIds() As String = GetSelectedSeriesIds()

        'tell the source to prepare search results for the the current selection
        source.PrepareBrowserSelection(selectedSeriesIds)
        Return ((Not selectedSeriesIds Is Nothing) AndAlso selectedSeriesIds.Count() > 0)
    End Function

    'make Enter key open an item
    Private Sub ChildListView_KeyUp(sender As Object, e As Windows.Forms.KeyEventArgs) Handles ChildListView.KeyUp
        If (e.KeyCode = Windows.Forms.Keys.Enter) Then
            ChildListView_Click(sender, Nothing)
        End If
    End Sub

    'process a click on the Open command on the right mouse button menu
    Private Sub OpenToolStripMenuItem_Click(sender As Object, e As EventArgs) Handles OpenToolStripMenuItem.Click, OpenToolStripMenuItem2.Click
        If (ChildListView.SelectedItems.Count > 0) Then
            Dim item As Windows.Forms.ListViewItem = ChildListView.SelectedItems.Item(0)
            Dim val = item.Name
            If (val = "..") Then
                'special 'parent' folder
                LoadCategory(parentId)
            ElseIf (IsNumeric(val)) Then
                'other folder
                LoadCategory(CInt(val))
            Else
                'notify database class of user selection
                If (PrepareCurrentSelection()) Then

                    'send message to EViews to preview the selected items
                    SendMessage(EViewsEdx.MessageId.PreviewSelected)
                End If
            End If
        End If
    End Sub

    'process a click on the Copy command on the right mouse button menu
    Private Sub CopyToolStripMenuItem_Click(sender As Object, e As EventArgs) Handles CopyToolStripMenuItem.Click
        If (ChildListView.SelectedItems.Count > 0) Then
            'notify database class of user selection
            If (PrepareCurrentSelection()) Then

                'send message to EViews to copy the selected items to the clipboard
                SendMessage(EViewsEdx.MessageId.CopySelected)
            End If
        End If
    End Sub


    'process a click on the Download command on the right mouse button menu
    Private Sub DownloadBulkFileToolStripMenuItem_Click(sender As Object, e As EventArgs) Handles DownloadBulkFileToolStripMenuItem.Click
        If (ChildListView.SelectedItems.Count > 0) Then
            Dim item As Windows.Forms.ListViewItem = ChildListView.SelectedItems.Item(0)
            Dim val = item.Name
            If (IsNumeric(val)) Then
                'tell source to download the content for this category
                Me.Cursor = System.Windows.Forms.Cursors.WaitCursor
                source.DownloadCategoryContent(CInt(val))
                Me.Cursor = System.Windows.Forms.Cursors.Default
            End If
        End If
    End Sub

    'timer used to delay initial setting of focus
    Private Sub Timer1_Tick(sender As Object, e As EventArgs) Handles Timer1.Tick
        Timer1.Enabled = False
        ChildListView.Focus()
    End Sub

    'class for handling sort order of items in list view
    Public Class CategoryBrowserItemComparer
        Implements IComparer

        'Public Sub New()
        'End Sub

        Public Function Compare(x As Object, y As Object) As Integer Implements IComparer.Compare
            Dim itemx As Windows.Forms.ListViewItem = x
            Dim itemy As Windows.Forms.ListViewItem = y
            If (itemx.Name = "..") Then
                'parent link always comes first
                Return -1
            ElseIf (itemy.Name = "..") Then
                'parent link always comes first
                Return 1
            ElseIf (IsNumeric(itemx.Name) And IsNumeric(itemy.Name)) Then
                'sort categories in alphabetical order
                Return StrComp(itemx.Text, itemy.Text, CompareMethod.Text)
            ElseIf (IsNumeric(itemx.Name) And Not IsNumeric(itemy.Name)) Then
                'categories should appear above series
                Return -1
            ElseIf (IsNumeric(itemy.Name) And Not IsNumeric(itemx.Name)) Then
                'categories should appear above series
                Return 1
            Else
                'sort series in alphabetical order
                Return StrComp(itemx.Text, itemy.Text, CompareMethod.Text)
            End If
        End Function
    End Class


End Class

