﻿'Manager class for the EIA online database
'
'A single instance of this manager class is constructed by EViews the first time the EIA online database is used.
'Only one manager class is constructed per session.
'

Imports System.Runtime.InteropServices

<ProgId(ServerDatabaseManager.myProgId),
        ClassInterface(ClassInterfaceType.None), _
        ComVisible(True)> _
Public Class ServerDatabaseManager
    Implements EViewsEdx.IDatabaseManager

    'data members of class
    Public Const myProgId As String = "EdxEia.ServerDatabaseManager"
    Private defaultServerUrl As String = "api.eia.gov"
    Private defaultApiKey As String = Nothing

    'called during EViews shutdown - nothing to do for this manager
    Public Sub Close() Implements EViewsEdx.IDatabaseManager.Close

    End Sub

    'this function is called by EViews immediately after the manager is created to notify    
    'the database manager of any user preferences that have been saved by EViews.
    '
    'the EIA online database uses this to save the API registration key required by the EIA
    '
    Public Sub SetPreferences(prefs As String) Implements EViewsEdx.IDatabaseManager.SetPreferences
        Dim splitPrefs = prefs.Split(";")
        For Each item In splitPrefs
            'pull off next preference and split to name and value
            item = Trim(item)
            Dim offset = InStr(item, ":")
            Dim name As String
            Dim value As String
            If (offset >= 0) Then
                name = Trim(Left(item, offset - 1))
                value = Trim(Mid(item, offset + 1))
            Else
                name = item
                value = Nothing
            End If

            If (name = "ApiKey") Then
                'if the preference name is 'ApiKey' save the key value into a member variable
                defaultApiKey = value
            End If
        Next
    End Sub

    'this function is called by EViews when a user selects the View...Preferences... menu item from
    'the EViews database window.
    '
    'note that any preferences settings returned to EViews in the prefs argument will be saved into the
    'EViews ini file and passed back to this class the next time it is created using the SetPreferences function
    '
    'for the EIA online database, we use this to provide an interface where the user can configure the API key.
    '
    Public Function ConfigurePreferences(server As String, username As String, password As String, ByRef prefs As String) As Boolean Implements EViewsEdx.IDatabaseManager.ConfigurePreferences
        'open the preferences dialog 
        Dim dialog As ServerPreferencesDialog = New ServerPreferencesDialog
        dialog.ApiKeyTextBox.Text = defaultApiKey
        If (dialog.ShowDialog() = Windows.Forms.DialogResult.OK) Then
            'create a new preferences string containing the API key
            defaultApiKey = dialog.ApiKeyTextBox.Text
            prefs = "ApiKey:" & Trim(defaultApiKey)
            Return True
        End If

        Return False
    End Function

    'this function is called by EViews once after this class is created to retrieve important information
    'about how EViews should interact with databases in this format.
    '
    'note that EViews passes in a client information string which can be used to check EViews version information.
    '
    'EViews also passes in proxy server information if the user has configured proxy settings within EViews. 
    'this information should generally be used by any web connections made by a client.
    '
    Public Function GetAttributes(clientInfo As String) As Object Implements EViewsEdx.IDatabaseManager.GetAttributes

        'parse client info
        Dim appName As String = Nothing
        Dim appVersion As Double
        Dim buildDate As Date
        Dim opts As Hashtable = Nothing
        ParseClientInfo(appName, appVersion, buildDate, opts, clientInfo)

        'check for version 8 of EViews
        If (appName <> "EViews" Or appVersion < 8) Then
            MsgBox("The EIA Database Extension requires version 8.1 of EViews.", MsgBoxStyle.Information)
            Throw New COMException("Unable to initialize the EIA Database Extension", EViewsEdx.ErrorCode.FOREIGN_INITIALISE_FAILED)
        End If

        'check for build date
        If (buildDate < "2014-09-15") Then
            MsgBox("The EIA Database Extension requires a newer build of EViews. Please use Help/EViews Update from your EViews menus to update to the latest build.", MsgBoxStyle.Information)
            Throw New COMException("Unable to initialize the EIA Database Extension", EViewsEdx.ErrorCode.FOREIGN_INITIALISE_FAILED)
        End If

        'configure proxy server (if options provided)
        SetProxyServer(opts, False)

        'return attributes
        Return "name=EIA Online, " & _
            "description=EIA Online Database, " & _
            "type=eia, " & _
            "login=server, " & _
            "server=" & defaultServerUrl & "," & _
            "dbid=eia," & _
            "nocreate, " & _
            "readonly, " & _
            "search=browser, " & _
            "searchattr=name," & _
            "dotnet=35"
    End Function

    'create a new database object and return it to EViews
    Public Function OpenDb(databaseId As String, oc_mode As EViewsEdx.OpenCreateMode, rw_mode As EViewsEdx.ReadWriteMode, _
                            server As String, username As String, password As String) As EViewsEdx.IDatabase Implements EViewsEdx.IDatabaseManager.OpenDb
        'check for API key
        While (Len(defaultApiKey) = 0)
            'if there is no API key - force configuration.
            '
            'this exception will make EViews call the ConfigurePreferences() function before attempting
            'to open this database again
            '
            Throw New COMException("", EViewsEdx.ErrorCode.FOREIGN_CONFIGURATION_REQUIRED)
        End While

        'create new database
        username = defaultApiKey
        Return New ServerDatabase(Me, databaseId, server, username, password)
    End Function

    Public Function GetDatabaseIds(server As String, username As String, password As String) As Object Implements EViewsEdx.IDatabaseManager.GetDatabaseIds
        Return Nothing
    End Function

    Public Function RenameDb(srcDatabaseId As String, destDatabaseId As String, server As String, username As String, password As String) As Boolean Implements EViewsEdx.IDatabaseManager.RenameDb
        Return False
    End Function

    Public Function CopyDb(srcDatabaseId As String, destDatabaseId As String, server As String, username As String, password As String, overwrite As Boolean) As Boolean Implements EViewsEdx.IDatabaseManager.CopyDb
        Return False
    End Function

    Public Function DeleteDb(databaseId As String, server As String, username As String, password As String) As Boolean Implements EViewsEdx.IDatabaseManager.DeleteDb
        Return False
    End Function

    'called by EViews when the name of an object returned by this database is illegal in the destination format
    '
    'this function provides the manager with an opportunity to suggest a modification to the name.
    '
    'for EIA databases, we trim off the trailing frequency if it matches the workfile frequency.
    'we also replace '-' characters with $ since the $ is unused by EIA but legal in EViews.
    '
    Public Function ProposeName(ByRef objectId As String, destFormat As EViewsEdx.DbFormat, destFreqInfo As String) As Boolean Implements EViewsEdx.IDatabaseManager.ProposeName
        If (destFormat = EViewsEdx.DbFormat.EViewsWorkfile Or destFormat = EViewsEdx.DbFormat.EViewsDatabase) Then
            'if fetching to the same frequency workfile, trim the frequency off the end of the name
            Dim destFields As String() = Split(destFreqInfo, " ")
            Dim destFreq As String = destFields(0)
            If (objectId.EndsWith("." & destFreq)) Then
                objectId = Left(objectId, Len(objectId) - Len(destFreq) - 1)
            End If

            'replace dot with underscore
            objectId = objectId.Replace("."c, "_"c)
            'replace minus sign with $
            objectId = objectId.Replace("-"c, "$"c)
            'don't prompt
            Return False
        End If
        Return False

    End Function

    Public Sub AdjustSearchNamePattern(ByRef objectIdPattern As String) Implements EViewsEdx.IDatabaseManager.AdjustSearchNamePattern

    End Sub

    'parse the client info string passed into GetDescription() by EViews
    Public Shared Sub ParseClientInfo(ByRef appName As String, ByRef appVersion As Double, ByRef buildDate As Date, ByRef opts As Hashtable, _
                                        ByVal clientInfo As String)
        Dim splitClientInfo = clientInfo.Split(" ")

        'get app name and version
        If (splitClientInfo.Count >= 1) Then
            Dim fullAppName As String = splitClientInfo(0)
            Dim offset As Integer = Len(fullAppName)
            appVersion = -1
            While (offset > 0)
                Dim version As String = fullAppName.Substring(offset - 1)
                If (IsNumeric(version)) Then
                    appName = Left(fullAppName, offset - 1)
                    appVersion = version
                End If
                offset = offset - 1
            End While
        Else
            appName = Nothing
            appVersion = -1
        End If

        'get build date
        If (splitClientInfo.Count >= 2) Then
            buildDate = splitClientInfo(1)      'should be in ISO format
        Else
            buildDate = Nothing
        End If

        'get options: format is a comma separated list of simple keywords or keyword=value pairs.
        opts = New Hashtable
        If (splitClientInfo.Count >= 3) Then
            Dim options As String = splitClientInfo(2)
            If (options.Length > 2 And Left(options, 1) = "(" And Right(options, 1) = ")") Then
                options = Mid(options, 2, Len(options) - 2)
                Dim optionsList() As String = options.Split(",")
                For Each optionSetting In optionsList
                    optionSetting = Trim(optionSetting)
                    Dim equalsOffset = optionSetting.IndexOf("=")
                    Dim key As String
                    Dim value As String
                    If (equalsOffset >= 0) Then
                        key = Trim(Left(optionSetting, equalsOffset))
                        value = Trim(Mid(optionSetting, equalsOffset + 2))
                    Else
                        key = optionSetting
                        value = Nothing
                    End If
                    opts.Add(key, value)
                Next
            End If
        End If
    End Sub

    'set up default proxy server info based on the EViews global option settings
    Public Function SetProxyServer(ByRef opts As Hashtable, ByVal useSSL As Boolean)
        If (Not opts("proxyserver") Is Nothing) Then
            Dim proxyUser As String = opts("proxyuser")
            Dim proxyPassword As String = opts("proxypwd")

            Dim protocol As String
            Dim proxyServer As String
            Dim defaultPort As Integer
            If (Not useSSL Or opts("proxyserverssl") Is Nothing) Then
                'regular
                protocol = "http"
                defaultPort = 80
                proxyServer = opts("proxyserver")
            Else
                'https
                protocol = "https"
                defaultPort = 443
                proxyServer = opts("proxyserverssl")
            End If

            'add default port if none provided
            If (Not proxyServer.Contains(":"c)) Then
                proxyServer = proxyServer & ":" & defaultPort
            End If

            'set proxy server info
            Dim credentials As System.Net.NetworkCredential = Nothing
            If (Not proxyUser Is Nothing) Then
                If (Not proxyUser.Contains("\"c)) Then
                    credentials = New System.Net.NetworkCredential(proxyUser, proxyPassword)
                Else
                    Dim slashOffset = proxyUser.IndexOf("\"c)
                    Dim proxyDomain As String = Left(proxyUser, slashOffset)
                    proxyUser = Mid(proxyUser, slashOffset + 2)
                    credentials = New System.Net.NetworkCredential(proxyUser, proxyPassword, proxyDomain)
                End If
            End If

            'set up default proxy server information
            Dim proxyHost As String = protocol & "://" & proxyServer & "/"
            If (Not credentials Is Nothing) Then
                System.Net.WebRequest.DefaultWebProxy = New System.Net.WebProxy(proxyHost, False, Nothing, credentials)
            Else
                System.Net.WebRequest.DefaultWebProxy = New System.Net.WebProxy(proxyHost, False, Nothing)
            End If
            Return True
        Else
            'no proxy server info
            Return False
        End If
    End Function

    'provide API key from current user preference settings
    Public Function GetApiKey()
        Return defaultApiKey
    End Function

End Class
