﻿'Database class for EIA Bulk File database
'
'Handles the fetching of series and category information from a local file in EIA bulk file format
'

'for COMException
Imports System.Runtime.InteropServices

Public Class FileDatabase
    Implements EViewsEdx.IDatabase
    Implements CategoryBrowserControl.ICategorySource

    Private tokenizer As EViewsEdx.JsonReader
    Private parser As EiaParser

    Private duplicateSeriesCount As Integer = 0
    Private duplicateCategoryCount As Integer = 0

    'used to remember what category browser was viewing when it was last open
    Private browserStartupCategory As Integer = -1

    'enumerator for erturnign search results
    Private searchEnumerator As IDictionaryEnumerator = Nothing

    'called by the database manager class when a new database is opened
    Public Sub New(ByVal filePath As String)
        MyBase.New()

        'create deserialzier
        tokenizer = New EViewsEdx.JsonReader

        'attach deserializer to EIA bulk file
        tokenizer.AttachToFile(filePath)

        'create EIA parser
        parser = New EiaParser(tokenizer)

        'scan entire bulk file for series and category objects
        parser.ScanBulkFile(True, duplicateSeriesCount, duplicateCategoryCount)
    End Sub

    'called by EViews immediately before the database object is deleted
    Public Sub Close() Implements EViewsEdx.IDatabase.Close
        'discard EIA parser
        parser = Nothing

        'free file lock
        tokenizer.Detach()
        tokenizer = Nothing

        'encourage .NET to free up memory 
        GC.Collect()
    End Sub

    'called by EViews when user clicks on View... Database Statistics menu item
    '
    'for EIA bulk files, we report the number of series and categories found and
    'also report information on any duplicates that were found in the file
    '
    Public Function GetAttributes() As Object Implements EViewsEdx.IDatabase.GetAttributes
        Dim seriesCount As Integer
        Dim categoryCount As Integer
        parser.GetCachedObjectCounts(seriesCount, categoryCount)

        Dim info = seriesCount & " series found" & vbCrLf & categoryCount & " categories found"

        If (duplicateSeriesCount > 0) Then
            info &= vbCrLf & vbCrLf & duplicateSeriesCount & " duplicate series found"
        End If
        If (duplicateCategoryCount > 0) Then
            info &= vbCrLf & vbCrLf & duplicateCategoryCount & " duplicate categories found"
        End If

        Dim attr(0, 1) As Object
        attr(0, 0) = "description"
        attr(0, 1) = info
        Return attr
    End Function

    Public Sub SetAttributes(attr As Object) Implements EViewsEdx.IDatabase.SetAttributes

    End Sub

    'called by EViews when the user fetches or copies a series from the database
    Public Sub ReadObject(objectId As String, destFreqInfo As String, _
                          ByRef attr As Object, ByRef vals As Object, ByRef ids As Object) Implements EViewsEdx.IDatabase.ReadObject

        If (Not parser.ReadSeriesInBulkFile(objectId, attr, vals, ids)) Then
            'series not found
            Throw New COMException("", EViewsEdx.ErrorCode.RECORD_NAME_INVALID)
        End If

    End Sub

    Public Sub ReadObjectAttributes(objectId As String, destFreqInfo As String, ByRef attr As Object) Implements EViewsEdx.IDatabase.ReadObjectAttributes
        If (Not parser.ReadSeriesAttributesInBulkFile(objectId, attr)) Then
            'series not found
            Throw New COMException("", EViewsEdx.ErrorCode.RECORD_NAME_INVALID)
        End If
    End Sub

    Public Sub ReadObjects(objectIds As Object, destFreqInfo As Object, ByRef attr As Object, ByRef vals As Object, ByRef ids As Object) Implements EViewsEdx.IDatabase.ReadObjects

    End Sub

    Public Sub SearchByAttributes(searchExpression As String, attrNames As String) Implements EViewsEdx.IDatabase.SearchByAttributes
        searchEnumerator = parser.GetCachedSeriesEnumerator()

    End Sub

    Public Function SearchNext(ByRef objectId As String, ByRef attr As Object) As Boolean Implements EViewsEdx.IDatabase.SearchNext
        If (searchEnumerator.MoveNext()) Then
            objectId = searchEnumerator.Key
            Dim seriesInfo As EiaSeriesInfo = searchEnumerator.Value

            'create attribute array for EViews
            attr = New Object(4, 1) {}  '5 by 2 array
            attr(0, 0) = "description"
            attr(0, 1) = seriesInfo.description
            attr(1, 0) = "freq"
            attr(1, 1) = seriesInfo.freq
            attr(2, 0) = "start"
            attr(2, 1) = seriesInfo.startText
            attr(3, 0) = "end"
            attr(3, 1) = seriesInfo.endText
            attr(4, 0) = "last_update"
            attr(4, 1) = seriesInfo.lastUpdate

            Return True
        Else
            searchEnumerator = Nothing
            Return False
        End If

    End Function

    'called by EViews when user aborts a search (hits the Escape key)
    Public Sub SearchAbort() Implements EViewsEdx.IDatabase.SearchAbort
        searchEnumerator = Nothing
    End Sub

    'called by EViews when user clicks on the 'Browse' toolbar button.
    'returns a custom browser control allowing the user to search through this database using the category tree.
    Public Function SearchByBrowser(browserArgs As Object, ByRef attrNames As String) As Object Implements EViewsEdx.IDatabase.SearchByBrowser
        attrNames = "name,description,freq,last_update"
        Return New CategoryBrowserControl(Me, browserStartupCategory)
    End Function

    Public Sub PrepareBrowserSelection(selectedIds() As String) Implements CategoryBrowserControl.ICategorySource.PrepareBrowserSelection
        searchEnumerator = parser.GetSelectedSeriesEnumerator(selectedIds)
    End Sub

    Public Sub ListObjectAttributes(ByRef attributeList As String, delim As String, ByRef scanForCustom As Boolean) Implements EViewsEdx.IDatabase.ListObjectAttributes

    End Sub

    Public Sub WriteObject(ByRef objectId As String, attr As Object, vals As Object, ids As Object, overwriteMode As EViewsEdx.WriteType) Implements EViewsEdx.IDatabase.WriteObject

    End Sub

    Public Sub WriteObjects(ByRef errors As Object, ByRef objectIds As Object, attr As Object, vals As Object, ids As Object, overwriteMode As EViewsEdx.WriteType) Implements EViewsEdx.IDatabase.WriteObjects

    End Sub

    Public Sub CopyObject(srcObjectId As String, ByRef destObjectId As String, Optional overwrite As Boolean = False) Implements EViewsEdx.IDatabase.CopyObject

    End Sub

    Public Sub DeleteObject(objectId As String) Implements EViewsEdx.IDatabase.DeleteObject

    End Sub

    Public Sub BeginWrite(label As String) Implements EViewsEdx.IDatabase.BeginWrite

    End Sub

    Public Sub EndWrite(reserved As Integer) Implements EViewsEdx.IDatabase.EndWrite

    End Sub

    Public Sub RenameObject(srcObjectId As String, destObjectId As String) Implements EViewsEdx.IDatabase.RenameObject

    End Sub

    Public Function GetCommandIds() As Object Implements EViewsEdx.IDatabase.GetCommandIds
        Return Nothing
    End Function

    Public Function DoCommand(commandId As String, args As Object) As Object Implements EViewsEdx.IDatabase.DoCommand
        Return Nothing
    End Function

    'callback used by the browser to retrieve parent/child information for the specified category
    '
    'browser uses categoryId < 0 to request info on the root category (true category number will be returned by this function)
    '
    Public Function GetCategoryInfo(id As Integer, ByRef parentId As Integer, ByRef description As String, ByRef categoryPath As String, _
                                    ByRef childIds() As Integer, ByRef childDescriptions() As String, _
                                    ByRef childSeriesNames() As String, ByRef childSeriesDescriptions() As String) As Integer Implements CategoryBrowserControl.ICategorySource.GetCategoryInfo
        If (id < 0) Then
            id = parser.GetRootCategory()
        End If

        'read info from cache
        parser.GetCategoryInfo(id, parentId, description, categoryPath, childIds, childDescriptions, childSeriesNames, childSeriesDescriptions)

        If (parser.GetCategoryState(parentId) < 0) Then
            'if parent id doesn't exist, change it to -1 to indicate there is not (usable) parent
            parentId = -1
        End If

        Return id
    End Function

    'callback used by the browser to save the current category as the startup category when browser next opens
    Public Sub SaveStartupCategory(id As Integer) Implements CategoryBrowserControl.ICategorySource.SaveStartupCategory
        browserStartupCategory = id
    End Sub

    'shouldn't ever be called for offline database
    Public Sub DownloadCategoryContent(id As Integer) Implements CategoryBrowserControl.ICategorySource.DownloadCategoryContent

    End Sub
End Class


