﻿'Parsing related classes: used to decode EIA JSON format data
'
'The EIAParser class is the main class that parses all EIA JSON data.
'It also maintains a cache containing series and category information that
'has already been read. When used with the EIA online database, the cache
'contains information for categories that have been visited within the browser.
'For EIA bulk files, the cache contains information for all series and 
'categories that were found when scanning the file - although series content
'is not held within the cache (only the file location of each series).
'

'cached info for an EIA series
Public Class EiaSeriesInfo
    Public location As Int64    'location of series object in json text
    Public parent As Integer    'parent category: -1 indicates parent not yet known

    'cached information on series
    Public description As String
    Public freq As String
    Public startText As String
    Public endText As String
    Public lastUpdate As Date

    Public Sub New()
        location = -1
        parent = -1
    End Sub
End Class

'cached info for an EIA category
Public Class EiaCategoryInfo
    Public location As Int64    'location of category object in json text
    Public parent As Integer    'parent category: negative value indicates topmost node

    'cached information on category
    Public description As String
    Public notes As String

    Public Sub New()
        location = -1
        parent = -1
    End Sub
End Class

'main class for parsing EIA JSON format data
Public Class EiaParser
    'input to the parser
    Private j As EViewsEdx.JsonReader

    'info on last token read
    Private t As EViewsEdx.JsonTokenType
    Private val As Object = Nothing

    'info on last object read
    Public lastObjectType As String = Nothing

    'error message returned by server when a web request fails
    Public lastErrorMessage As String = Nothing

    'buffers for data values and ids of last series read
    Dim cachedSeriesId As String = Nothing

    Dim seriesAttributeBufferSize As Integer = -1 'size of attribute buffer arrays
    Dim seriesAttributeBufferNames() As String  'names of attributes
    Dim seriesAttributeBufferVals() As Object   'values of attributes
    Dim seriesAttributeBufferCount As Integer = -1 'how many attributes currently in the buffer

    Dim seriesObsBufferSize As Integer = -1 'size of observation buffer arrays
    Dim seriesObsBufferIds() As String = Nothing
    Dim seriesObsBufferVals() As Double = Nothing
    Dim seriesObsBufferCount As Integer = -1 'how many attributes currently in the buffer

    'info from last category read
    Public lastCategoryId As Integer

    'series information cached from scanning bulk file or previously fetched from server
    Dim cachedSeriesInfo As Hashtable = Nothing

    'category information cached from scanning bulk file or previously fetched from server
    Dim cachedCategoryInfo As Hashtable = Nothing

    'landing point to catch unusual events (only if debugging)
    Public Sub DebugBreak()
#If DEBUG Then
        'Stop
#End If
    End Sub

    'create the parser and attach it to the EViews json tokenizer class
    Public Sub New(jsonReader As EViewsEdx.JsonReader)
        j = jsonReader

        cachedSeriesInfo = New Hashtable
        cachedCategoryInfo = New Hashtable
    End Sub

    'bring in the next input token (and read past it)
    Private Sub ReadToken()
        t = j.ReadToken(val)
        If (t = EViewsEdx.JsonTokenType.JSON_ERROR) Then
            DebugBreak()
        End If
    End Sub

    'bring in the next input token (and don't read past it)
    Private Sub PeekToken()
        t = j.PeekToken(val)
        If (t = EViewsEdx.JsonTokenType.JSON_ERROR) Then
            DebugBreak()
        End If
    End Sub

    'top level function for parsing an entire EIA bulk file.
    'file locations for all series and categories will be cached during this function.
    Public Function ScanBulkFile(skipSeriesValues As Boolean, _
                                 ByRef duplicateSeriesCount As Integer, ByRef duplicateCategoryCount As Integer) As Boolean
        cachedSeriesInfo.Clear()
        cachedCategoryInfo.Clear()

        Dim seriesCount As Integer = 0
        Dim categoryCount As Integer = 0

        'loop over series objects.
        'these are simply spooled one after another at the top of the file.
        duplicateSeriesCount = 0
        While True
            PeekToken()
            If (t = EViewsEdx.JsonTokenType.JSON_EOF) Then Exit While

            If (t <> EViewsEdx.JsonTokenType.JSON_OBJECT_BEGIN) Then Return False

            Dim seriesStart As Int64 = j.GetPosition()
            Dim seriesId As String = Nothing
            Dim seriesInfo = New EiaSeriesInfo
            seriesInfo.location = seriesStart
            If (Not ReadSeries(seriesId, skipSeriesValues, seriesInfo)) Then Exit While

            'try/catch to look for duplicates
            Try
                cachedSeriesInfo.Add(seriesId, seriesInfo)
            Catch
                'duplicate series in file!?
                duplicateSeriesCount += 1
            End Try

            seriesCount += 1
        End While

        If (duplicateSeriesCount > 0) Then
            'duplicate series objects found (same series_id)
            DebugBreak()
        End If

        'read categories (at bottom of bulk file)
        duplicateCategoryCount = 0
        If (t <> EViewsEdx.JsonTokenType.JSON_EOF And t <> EViewsEdx.JsonTokenType.JSON_ERROR) Then

            'loop over category objects
            While True
                PeekToken()
                If (t = EViewsEdx.JsonTokenType.JSON_EOF) Then Exit While
                If (t <> EViewsEdx.JsonTokenType.JSON_OBJECT_BEGIN) Then Return False

                Dim categoryStart As Int64 = j.GetPosition()
                Dim categoryId As Integer = -1
                If (Not ReadCategory(categoryId)) Then
                    'not a category?
                    Exit While
                End If

                'add start position to category
                cachedCategoryInfo(categoryId).location = categoryStart

                categoryCount = categoryCount + 1
            End While
        End If

        If (duplicateCategoryCount > 0) Then
            'duplicate category objects found (same category_id)
            DebugBreak()
        End If

        If (t <> EViewsEdx.JsonTokenType.JSON_EOF) Then
            'didn't make it to end of file
            DebugBreak()
            Return False
        End If

        Return True
    End Function

    'fetch series information from a (previously scanned) bulk file into the series buffer
    Public Function CacheSeriesInBulkFile(id As String, skipSeriesdata As Boolean) As Boolean
        Dim info = cachedSeriesInfo(id)
        If (Not info Is Nothing) Then
            j.Restart(info.location)

            Dim readId As String = Nothing
            If (Not ReadSeries(readId, skipSeriesdata)) Then
                'series invalid / location bad?
                Return False
            End If

            Return True
        Else
            'name not found
            Return False
        End If

    End Function

    'package attributes of last series read into an EViews attribute array
    Public Sub FetchSeriesAttributesFromCache(ByRef attr As Object)
        attr = New Object(seriesAttributeBufferCount - 1, 1) {}
        For i = 0 To seriesAttributeBufferCount - 1
            attr(i, 0) = seriesAttributeBufferNames(i)
            attr(i, 1) = seriesAttributeBufferVals(i)
        Next
    End Sub

    'read series attributes only (not observations)
    Public Function ReadSeriesAttributesInBulkFile(id As String, ByRef attr As Object) As Boolean
        If (CacheSeriesInBulkFile(id, True)) Then
            'extract attributes from cache
            FetchSeriesAttributesFromCache(attr)
            Return True
        Else
            'name not found
            Return False
        End If
    End Function

    'package attributes and observations of last series read into EViews attrribute, values and ids arrays
    Public Sub FetchSeriesFromCache(ByRef attr As Object, ByRef vals As Object, ByRef ids As Object)
        'extract attributes from cache
        FetchSeriesAttributesFromCache(attr)

        'extract observations from cache into EViews ids and vals arrays
        ids = New String(seriesObsBufferCount - 1) {}
        vals = New Double(seriesObsBufferCount - 1) {}
        For i = 0 To seriesObsBufferCount - 1
            ids(i) = seriesObsBufferIds(i)
            vals(i) = seriesObsBufferVals(i)
        Next

    End Sub

    'read all series info (attributes and observations)
    Public Function ReadSeriesInBulkFile(id As String, ByRef attr As Object, ByRef vals As Object, ByRef ids As Object) As Boolean
        'load series info into cache buffer
        If (CacheSeriesInBulkFile(id, False)) Then
            FetchSeriesFromCache(attr, vals, ids)
            Return True
        Else
            'name not found
            Return False
        End If
    End Function

    'returns an enumerator for all cached series
    Public Function GetCachedSeriesEnumerator() As IDictionaryEnumerator
        Return cachedSeriesInfo.GetEnumerator()
    End Function

    'returns an enumerator for cached series with the specified names
    Public Function GetSelectedSeriesEnumerator(ids() As String) As IDictionaryEnumerator
        If (Not ids Is Nothing) Then
            Dim selectedSeriesInfo As Hashtable = New Hashtable
            For Each id In ids
                selectedSeriesInfo.Add(id, cachedSeriesInfo(id))
            Next
            Return selectedSeriesInfo.GetEnumerator()
        Else
            Return Nothing
        End If
    End Function

    'parse a reply returned by the EIA Web server
    Public Function ReadWebReply() As Boolean
        'read (single) object contained in reply
        If (Not ReadWebObject()) Then Return False

        'check that we have reached the end of the input
        ReadToken()
        If (t <> EViewsEdx.JsonTokenType.JSON_EOF) Then Return False

        Return True

    End Function

    'parse an object returned by the EIA web server.
    'type of object seen will be saved into lastObjectType.
    Public Function ReadWebObject() As Boolean
        'look for object start
        ReadToken()
        If (t <> EViewsEdx.JsonTokenType.JSON_OBJECT_BEGIN) Then Return False

        'loop over fields
        While True
            ReadToken()
            If (t = EViewsEdx.JsonTokenType.JSON_OBJECT_END) Then Exit While
            If (t <> EViewsEdx.JsonTokenType.JSON_FIELDNAME) Then Return False 'eg. unanticipated eof

            If (val = "request") Then
                'contains input info: eg. {"command":"series","series_id":"sssssss"} 
                j.SkipValue()
            ElseIf (val = "series") Then
                'found series node
                lastObjectType = val
                If (Not ReadSeriesArray()) Then Return False
            ElseIf (val = "category") Then
                'found category node
                lastObjectType = val
                If (Not ReadCategory(lastCategoryId)) Then Return False
            ElseIf (val = "data") Then
                'error coding? data field value contains an object with field "error" containg an error message string
                lastObjectType = "error"
                If (Not ReadError(lastErrorMessage)) Then Return False
                DebugBreak()
            Else
                'unknown field
                j.SkipValue()
            End If
        End While

        Return True

    End Function

    'parse EIA web reply error info - stored in the 'data' field of the reply
    Public Function ReadError(ByRef errmsg As String) As Boolean

        'look for object start
        ReadToken()
        If (t <> EViewsEdx.JsonTokenType.JSON_OBJECT_BEGIN) Then Return False

        'loop over fields
        Dim hasError As Boolean = False
        While True
            ReadToken()
            If (t = EViewsEdx.JsonTokenType.JSON_OBJECT_END) Then Exit While
            If (t <> EViewsEdx.JsonTokenType.JSON_FIELDNAME) Then Return False 'eg. unanticipated eof

            If (val = "error") Then
                'error field - get value
                ReadToken()
                If (t <> EViewsEdx.JsonTokenType.JSON_STRING) Then Return False
                hasError = True
                errmsg = val
            Else
                'skip past other fields
                j.SkipValue()
            End If
        End While

        'check for required fields
        If (Not hasError) Then
            'if no error fielde found, probably not an error - rewind so that someone else has a chance to read this object
            j.RewindValue()
            Return False
        End If

        Return True
    End Function

    'parse an array of EIA series objects
    Public Function ReadSeriesArray() As Boolean
        'look for array start
        ReadToken()
        If (t <> EViewsEdx.JsonTokenType.JSON_ARRAY_BEGIN) Then Return False

        'loop over series objects
        While True
            PeekToken()
            If (t = EViewsEdx.JsonTokenType.JSON_ARRAY_END) Then Exit While

            Dim seriesId As String = Nothing
            If (Not ReadSeries(seriesId)) Then Return False
        End While

        'skip array end
        ReadToken()

        Return True
    End Function

    'parse a single EIA series object
    '
    'attribute names and values are saved into seriesAttributeBufferNames() and series attributeBufferVals()
    'observation ids and values are saved into seriesObsBufferIds() and seriesObsBufferVals()
    '
    Public Function ReadSeries(ByRef id As String, Optional skipSeriesData As Boolean = False, _
                               Optional ByRef seriesInfo As EiaSeriesInfo = Nothing) As Boolean
        'look for object start
        ReadToken()
        If (t <> EViewsEdx.JsonTokenType.JSON_OBJECT_BEGIN) Then Return False

        'make sure we have buffer arrays allocated for holding attribute names and values
        If (seriesAttributeBufferSize < 0) Then
            seriesAttributeBufferSize = 50
            seriesAttributeBufferNames = New String(seriesAttributeBufferSize - 1) {}
            seriesAttributeBufferVals = New Object(seriesAttributeBufferSize - 1) {}
        End If

        seriesAttributeBufferCount = -1    'set to invalid value - will change if we complete without error

        'loop over fields
        Dim attributeCount As Integer = 0
        Dim HasSeriesId As Boolean = False
        Dim HasValues As Boolean = False
        Dim description As String = Nothing
        Dim freqText As String = Nothing
        Dim startText As String = Nothing
        Dim endText As String = Nothing
        Dim lastUpdate As Date = Nothing
        While True
            ReadToken()
            If (t = EViewsEdx.JsonTokenType.JSON_OBJECT_END) Then Exit While
            If (t <> EViewsEdx.JsonTokenType.JSON_FIELDNAME) Then Return False 'eg. unanticipated eof

            If (attributeCount >= seriesAttributeBufferSize) Then
                'need bigger buffers - double in size
                seriesAttributeBufferSize = seriesAttributeBufferSize * 2
                ReDim Preserve seriesAttributeBufferNames(seriesAttributeBufferSize)
                ReDim Preserve seriesAttributeBufferVals(seriesAttributeBufferSize)
            End If

            If (val = "series_id") Then
                'identifier - string value
                HasSeriesId = True
                ReadToken()
                If (t <> EViewsEdx.JsonTokenType.JSON_STRING) Then Return False
                id = val
            ElseIf (val = "description") Then
                ReadToken()
                If (t <> EViewsEdx.JsonTokenType.JSON_STRING) Then Return False
                'seriesAttributeBufferNames(attributeCount) = "remarks" 'translate attribute name for EViews
                'seriesAttributeBufferVals(attributeCount) = val
                'attributeCount = attributeCount + 1
            ElseIf (val = "f") Then
                'found frequency - string value
                ReadToken()
                If (t <> EViewsEdx.JsonTokenType.JSON_STRING) Then Return False
                freqText = val
                If (freqText = "4") Then
                    'translate '4' frequency (four week moving average?) to weekly
                    freqText = "W"
                End If
                seriesAttributeBufferNames(attributeCount) = "freq" 'translate attribute name for EViews
                seriesAttributeBufferVals(attributeCount) = freqText
                attributeCount = attributeCount + 1
            ElseIf (val = "start") Then
                'found start attribute - save for processing outside of loop so freq is available (even if fields are out of order)
                ReadToken()
                If (t <> EViewsEdx.JsonTokenType.JSON_STRING) Then Return False
                startText = val
            ElseIf (val = "end") Then
                'found end attribute - save for processing outside of loop so freq is available (even if fields are out of order)
                ReadToken()
                If (t <> EViewsEdx.JsonTokenType.JSON_STRING) Then Return False
                endText = val
            ElseIf (val = "update" Or val = "updated" Or val = "last_updated") Then
                'update seems to appear a few different ways - change to EViews last_update
                ReadToken()
                If (t <> EViewsEdx.JsonTokenType.JSON_STRING) Then Return False
                lastUpdate = val
                seriesAttributeBufferNames(attributeCount) = "last_update" 'translate attribute name for EViews
                seriesAttributeBufferVals(attributeCount) = val
                attributeCount = attributeCount + 1
            ElseIf (val = "name") Then
                'not object name - actually a one line description?
                ReadToken()
                If (t <> EViewsEdx.JsonTokenType.JSON_STRING) Then Return False
                description = val
                seriesAttributeBufferNames(attributeCount) = "description" 'translate attribute name for EViews
                seriesAttributeBufferVals(attributeCount) = val
                attributeCount = attributeCount + 1
            ElseIf (val = "data") Then
                'data array
                HasValues = True
                If (Not skipSeriesData) Then
                    If (Not ReadSeriesObservations()) Then Return False
                Else
                    j.SkipValue()
                End If
            Else
                'miscellaneous attribute
                seriesAttributeBufferNames(attributeCount) = val
                PeekToken()
                If (t = EViewsEdx.JsonTokenType.JSON_STRING Or t = EViewsEdx.JsonTokenType.JSON_NUMBER _
                        Or t = EViewsEdx.JsonTokenType.JSON_BOOL Or t = EViewsEdx.JsonTokenType.JSON_NULL) Then
                    'attribute is simply scalar value - return it
                    seriesAttributeBufferVals(attributeCount) = val
                    attributeCount = attributeCount + 1
                Else
                    'attribute is an array or object, ignore it (eg. "latlon" array)
                End If
                j.SkipValue()

            End If

        End While

        'check for required fields
        If (Not HasSeriesId) Then
            'if no series id found, probably not a series - rewind so that someone else has a chance to read this object
            j.RewindValue()
            Return False
        End If

        'handle start and end fields (which need to be translated based on frequency)
        If (Len(startText) > 0) Then
            'fixup for unusual EIA start and end date formats (YYYYPP?)
            If (freqText = "M" And InStr(startText, "M") <= 0) Then
                'make EIA monthly format YYYYMM explicit
                startText = Left(startText, 4) & "M" & Mid(startText, 5)
            ElseIf (freqText = "Q" And InStr(startText, "Q") <= 0) Then
                'make EIA monthly format YYYY0Q explicit
                startText = Left(startText, 4) & "Q" & Right(startText, 1)
            ElseIf (Len(startText) = 8) Then
                'convert YYYYMMDD to YYYY-MM-DD
                startText = Left(startText, 4) & "-" & Mid(startText, 5, 2) & "-" & Right(startText, 2)
            End If
            seriesAttributeBufferNames(attributeCount) = "start"
            seriesAttributeBufferVals(attributeCount) = startText
            attributeCount = attributeCount + 1
        End If
        If (Len(endText) > 0) Then
            'fixup for unusual EIA start and end date formats
            If (freqText = "M" And InStr(endText, "M") <= 0) Then
                'make EIA monthly format YYYYMM explicit
                endText = Left(endText, 4) & "M" & Mid(endText, 5)
            ElseIf (freqText = "Q" And InStr(endText, "Q") <= 0) Then
                'make EIA monthly format YYYY0Q explicit
                endText = Left(endText, 4) & "Q" & Right(endText, 1)
            ElseIf (Len(endText) = 8) Then
                'convert YYYYMMDD to YYYY-MM-DD
                endText = Left(endText, 4) & "-" & Mid(endText, 5, 2) & "-" & Right(endText, 2)
            End If
            seriesAttributeBufferNames(attributeCount) = "end"
            seriesAttributeBufferVals(attributeCount) = endText
            attributeCount = attributeCount + 1
        End If
        'monthly data - also need to translate observation ids
        If (freqText = "M") Then
            For i = 0 To seriesObsBufferCount - 1
                'make EIA monthly format YYYYMM explicit
                Dim oldId As String = seriesObsBufferIds(i)
                seriesObsBufferIds(i) = Left(oldId, 4) & "M" & Mid(oldId, 5)
            Next
        End If

        'save fields into seriesInfo
        If (Not seriesInfo Is Nothing) Then
            seriesInfo.description = description
            seriesInfo.freq = freqText
            seriesInfo.startText = startText
            seriesInfo.endText = endText
            seriesInfo.lastUpdate = lastUpdate
        End If

        seriesAttributeBufferCount = attributeCount

        Return True

    End Function

    'parse the observations (data values and ids) of an EIA series object
    Public Function ReadSeriesObservations() As Boolean
        'look for array start
        ReadToken()
        If (t <> EViewsEdx.JsonTokenType.JSON_ARRAY_BEGIN) Then Return False

        'make sure we have buffer arrays allocated for holding values and ids
        If (seriesObsBufferSize < 0) Then
            seriesObsBufferSize = 1000
            seriesObsBufferIds = New String(seriesObsBufferSize - 1) {}
            seriesObsBufferVals = New Double(seriesObsBufferSize - 1) {}
        End If

        seriesObsBufferCount = -1    'set to invalid value - will change if we complete without error

        'loop over observations
        Dim observationCount As Integer = 0
        While (True)
            'read array begin
            ReadToken()
            If (t = EViewsEdx.JsonTokenType.JSON_ARRAY_END) Then Exit While
            If (t <> EViewsEdx.JsonTokenType.JSON_ARRAY_BEGIN) Then Return False

            If (observationCount >= seriesObsBufferSize) Then
                'need bigger buffer - double in size
                seriesObsBufferSize = seriesObsBufferSize * 2
                ReDim Preserve seriesObsBufferIds(seriesObsBufferSize)
                ReDim Preserve seriesObsBufferVals(seriesObsBufferSize)
            End If

            'read observation id
            ReadToken()
            If (t <> EViewsEdx.JsonTokenType.JSON_STRING) Then Return False
            seriesObsBufferIds(observationCount) = val

            'read value
            ReadToken()
            If (t = EViewsEdx.JsonTokenType.JSON_NUMBER) Then
                'number
                seriesObsBufferVals(observationCount) = val
            ElseIf (t = EViewsEdx.JsonTokenType.JSON_BOOL) Then
                'boolean values
                seriesObsBufferVals(observationCount) = val
            ElseIf (t = EViewsEdx.JsonTokenType.JSON_STRING) Then
                'string values - treat as NA
                seriesObsBufferVals(observationCount) = Double.NaN
            ElseIf (t = EViewsEdx.JsonTokenType.JSON_NULL) Then
                'null values -- treat as NA
                seriesObsBufferVals(observationCount) = Double.NaN
            Else
                'other token (object or array?)
                Return False
            End If

            'read array end
            ReadToken()
            If (t <> EViewsEdx.JsonTokenType.JSON_ARRAY_END) Then Return False

            observationCount = observationCount + 1
        End While

        seriesObsBufferCount = observationCount

        Return True
    End Function

    'parse an EIA category, saving results into cachedCategoryInfo and cachedSeriesInfo
    '
    'note that the bulk file does not contain childcategories (to avoid duplication?).
    'non-terminal nodes are simply listed with their parent_category_id values.
    '
    Public Function ReadCategory(ByRef categoryId As Integer) As Boolean
        'look for object start
        ReadToken()
        If (t <> EViewsEdx.JsonTokenType.JSON_OBJECT_BEGIN) Then Return False

        'loop over fields
        Dim hasCategoryId As Boolean = False
        Dim parentId As Integer = -1
        Dim description As String = Nothing
        Dim notes As String = Nothing
        While True
            ReadToken()
            If (t = EViewsEdx.JsonTokenType.JSON_OBJECT_END) Then Exit While
            If (t <> EViewsEdx.JsonTokenType.JSON_FIELDNAME) Then Return False 'eg. unanticipated eof

            If (val = "category_id") Then
                'string value -- although strange since category_ids in childcategories are integers
                hasCategoryId = True
                ReadToken()
                If (t <> EViewsEdx.JsonTokenType.JSON_STRING And t <> EViewsEdx.JsonTokenType.JSON_NUMBER) Then Return False
                categoryId = val
            ElseIf (val = "parent_category_id") Then
                'string value -- although strange since category_ids in childcategories are integers
                ReadToken()
                If (t = EViewsEdx.JsonTokenType.JSON_STRING Or t = EViewsEdx.JsonTokenType.JSON_NUMBER) Then
                    parentId = val
                ElseIf (t = EViewsEdx.JsonTokenType.JSON_NULL) Then
                    'use special coding of -1 for topmost node
                    parentId = -1
                Else
                    'invalid value
                    Return False
                End If
            ElseIf (val = "name") Then
                'short text description
                ReadToken()
                If (t <> EViewsEdx.JsonTokenType.JSON_STRING) Then Return False
                description = val
            ElseIf (val = "notes") Then
                ReadToken()
                If (t <> EViewsEdx.JsonTokenType.JSON_STRING) Then Return False
                notes = val
            ElseIf (val = "childcategories") Then
                'not used in bulk file (only in category request from Web)
                'note assumption here that category_id comes before childcategories (i.e. that fields are ordered)
                If (Not ReadChildCategoriesArray(categoryId)) Then Return False
            ElseIf (val = "childseries") Then
                'note assumption here that category_id comes before childseries (i.e. that fields are ordered)
                If (Not ReadChildSeriesArray(categoryId)) Then Return False
            Else
                j.SkipValue()
            End If
        End While

        'check for required fields
        If (hasCategoryId) Then
            'look for existing entry for this category
            Dim categoryInfo As EiaCategoryInfo = cachedCategoryInfo(categoryId)
            If (categoryInfo Is Nothing) Then
                'add this category to cached category table
                categoryInfo = New EiaCategoryInfo
                categoryInfo.location = 0 'mark category as visited
                categoryInfo.parent = parentId
                categoryInfo.description = description
                categoryInfo.notes = notes
                cachedCategoryInfo.Add(categoryId, categoryInfo)
            Else
                'category already in table
                categoryInfo.notes = notes
                If (categoryInfo.location < 0) Then
                    categoryInfo.location = 0  'mark category as visited
                End If
                If (parentId <> categoryInfo.parent Or description <> categoryInfo.description) Then
                    'new info doesn't match existing entry
                    DebugBreak()
                End If

            End If

            Return True
        Else
            Return False
        End If

    End Function

    'parse an array of EIA child categories
    Public Function ReadChildCategoriesArray(parentId As Integer) As Boolean
        'look for array start
        ReadToken()
        If (t <> EViewsEdx.JsonTokenType.JSON_ARRAY_BEGIN) Then Return False

        'loop over child categories
        Dim childCategoryCount As Integer = 0
        While True
            PeekToken()
            If (t = EViewsEdx.JsonTokenType.JSON_ARRAY_END) Then Exit While

            'read a child cateogry
            Dim childId As Integer = -1
            Dim childDescription As String = Nothing
            If (Not ReadChildCategory(childId, childDescription)) Then Return False

            'if child category hasn't been seen before, add it to cachedCategoryInfo table
            Dim categoryInfo = cachedCategoryInfo(childId)
            If (categoryInfo Is Nothing) Then
                categoryInfo = New EiaCategoryInfo
                categoryInfo.parent = parentId
                categoryInfo.description = childDescription
                cachedCategoryInfo.Add(childId, categoryInfo)
            End If

            childCategoryCount = childCategoryCount + 1
        End While

        'skip array end
        ReadToken()

        Return True
    End Function

    'parse a single EIA child category object
    Private Function ReadChildCategory(ByRef childId As Integer, ByRef childDescription As String) As Boolean
        'look for object start
        ReadToken()
        If (t <> EViewsEdx.JsonTokenType.JSON_OBJECT_BEGIN) Then Return False

        'loop over fields
        While True
            ReadToken()
            If (t = EViewsEdx.JsonTokenType.JSON_OBJECT_END) Then Exit While
            If (t <> EViewsEdx.JsonTokenType.JSON_FIELDNAME) Then Return False 'eg. unanticipated eof

            If (val = "category_id") Then
                'number -- although strange since category_id in category is an integer
                ReadToken()
                If (t <> EViewsEdx.JsonTokenType.JSON_STRING And t <> EViewsEdx.JsonTokenType.JSON_NUMBER) Then Return False
                childId = val
            ElseIf (val = "name") Then
                ReadToken()
                If (t <> EViewsEdx.JsonTokenType.JSON_STRING) Then Return False
                childDescription = val
            Else
                'unrecognized field
                j.SkipValue()
            End If
        End While

        Return True
    End Function

    'parse array containing names of child series
    '
    'note that in bulk file, this is a simple array of series names, while in a web request
    'this is an array of objects containing fields: "series_id", "name", "f", "units", and "updated"
    '
    Private Function ReadChildSeriesArray(Optional parentId As Integer = -999) As Boolean
        'look for object start
        ReadToken()
        If (t <> EViewsEdx.JsonTokenType.JSON_ARRAY_BEGIN) Then Return False

        'loop over child series
        Dim ChildSeriesCount As Integer = 0
        While True
            PeekToken()
            If (t = EViewsEdx.JsonTokenType.JSON_ARRAY_END) Then Exit While

            If (t = EViewsEdx.JsonTokenType.JSON_STRING) Then
                'child series name
                ReadToken()
                If (parentId >= 0) Then
                    'add parent category info to series
                    Dim currentSeriesInfo = cachedSeriesInfo(val)
                    If (Not currentSeriesInfo Is Nothing) Then
                        currentSeriesInfo.parent = parentId
                    Else
                        'couldn't find series 
                        DebugBreak()
                    End If
                End If
            ElseIf (t = EViewsEdx.JsonTokenType.JSON_OBJECT_BEGIN) Then
                'read series object
                Dim seriesid As String = Nothing
                Dim newSeriesInfo As EiaSeriesInfo = New EiaSeriesInfo
                If (Not ReadSeries(seriesid, True, newSeriesInfo)) Then Return False

                'do we already have an entry for this series?
                Dim currentSeriesInfo = cachedSeriesInfo(seriesid)
                If (Not currentSeriesInfo Is Nothing) Then
                    'if so, update its parent category
                    currentSeriesInfo.parent = parentId
                Else
                    'no existing entry for this series - add it
                    newSeriesInfo.parent = parentId
                    cachedSeriesInfo.Add(seriesid, newSeriesInfo)
                End If
            Else
                'element is neither a simple string name nor a series object?
                Return False
            End If

            ChildSeriesCount = ChildSeriesCount + 1
        End While

        'skip end of object token
        ReadToken()

        Return True
    End Function

    'find the category that is the root of all categories that
    'are currently in the cache.
    '
    'used with EIA bulk files to determine the starting category
    'for the browser.
    '
    Public Function GetRootCategory() As Integer

        'loop over all categories
        Dim categoryEnumerator = cachedCategoryInfo.GetEnumerator()
        Dim categoryInfo As EiaCategoryInfo = Nothing
        Dim rootId As Integer = -1
        Dim currentId As Integer
        Dim parentId As Integer
        While (categoryEnumerator.MoveNext())
            'move up to each category's parent until the parent doesn't exist
            parentId = categoryEnumerator.Key
            categoryInfo = categoryEnumerator.Value
            Do
                currentId = parentId
                parentId = categoryInfo.parent
                categoryInfo = cachedCategoryInfo(parentId)
            Loop While (Not categoryInfo Is Nothing)

            'check whether the root is the same for all categories
            If (rootId >= 0 And currentId <> rootId) Then
                'more than one root node found?
                DebugBreak()
            End If
            rootId = currentId

        End While

        Return rootId
    End Function

    'check the current state of a category. returns:
    '-1 if category doesn't exist
    ' 0 if cateogry exists, but doesn't yet have child info (category was created as a child of another category)
    ' 1 if category exists and has child info (category was loaded itself)
    '
    Public Function GetCategoryState(categoryid As Integer) As Integer
        Dim categoryInfo = cachedCategoryInfo(categoryid)
        If (Not categoryInfo Is Nothing) Then
            If (categoryInfo.location >= 0) Then
                Return 1
            Else
                Return 0
            End If
        Else
            Return -1
        End If
    End Function

    'retrieve the text description of a category from the category cache
    Public Function GetCategoryDescription(categoryId As Integer) As String
        Dim categoryInfo = cachedCategoryInfo(categoryId)
        If (Not categoryInfo Is Nothing) Then
            Return categoryInfo.description
        Else
            Return Nothing
        End If
    End Function

    'retrieve information about a category from the category and series caches
    Public Function GetCategoryInfo(categoryid As Integer, ByRef parentId As Integer, ByRef description As String, ByRef categoryPath As String, _
                                    ByRef childIds() As Integer, ByRef childDescriptions() As String, _
                                    ByRef childSeriesNames() As String, ByRef childSeriesDescriptions() As String) As Boolean
        Dim categoryInfo As EiaCategoryInfo = cachedCategoryInfo(categoryid)
        If (Not categoryInfo Is Nothing) Then
            parentId = categoryInfo.parent
            description = categoryInfo.description

            'look for child categories
            Dim childCategoryList As ArrayList = New ArrayList
            Dim categoryEnumerator = cachedCategoryInfo.GetEnumerator()
            While (categoryEnumerator.MoveNext())
                categoryInfo = categoryEnumerator.Value
                If (categoryInfo.parent = categoryid) Then
                    childCategoryList.Add(categoryEnumerator.Key)
                End If
            End While

            'fill child category arrays
            Dim childCategoryCount = childCategoryList.Count
            childIds = New Integer(childCategoryCount - 1) {}
            childDescriptions = New String(childCategoryCount - 1) {}
            Dim i As Integer = 0
            For Each child In childCategoryList
                childIds(i) = child
                categoryInfo = cachedCategoryInfo(child)
                childDescriptions(i) = categoryInfo.description
                i = i + 1
            Next

            'look for child series
            Dim seriesInfo As EiaSeriesInfo = Nothing
            Dim childSeriesList As ArrayList = New ArrayList
            Dim seriesEnumerator = cachedSeriesInfo.GetEnumerator()
            While (seriesEnumerator.MoveNext())
                seriesInfo = seriesEnumerator.Value
                If (seriesInfo.parent = categoryid) Then
                    childSeriesList.Add(seriesEnumerator.Key)
                End If
            End While

            'fill child series array
            Dim childSeriesCount = childSeriesList.Count
            childSeriesNames = New String(childSeriesCount - 1) {}
            childSeriesDescriptions = New String(childSeriesCount - 1) {}
            i = 0
            For Each child In childSeriesList
                childSeriesNames(i) = child
                seriesInfo = cachedSeriesInfo(child)
                childSeriesDescriptions(i) = seriesInfo.description
                i = i + 1
            Next

            categoryPath = GetCategoryPath(categoryid)

            Return True
        End If
        Return False
    End Function

    'return a descriptive path for a category all the way up to the (last available) root node
    Public Function GetCategoryPath(categoryId As Integer, Optional delim As String = " > ") As String

        'go up category tree until the parent category doesn't exist
        Dim path As String = Nothing
        Dim categoryInfo As EiaCategoryInfo = cachedCategoryInfo(categoryId)
        While (Not categoryInfo Is Nothing)
            'add current category to path
            If (path Is Nothing) Then
                path = categoryInfo.description
            Else
                path = categoryInfo.description & delim & path
            End If
            'move up to parent
            categoryInfo = cachedCategoryInfo(categoryInfo.parent)
        End While

        Return path
    End Function

    'return how many series / categories are currently cached
    Public Sub GetCachedObjectCounts(ByRef seriesCount As Integer, ByRef categoryCount As Integer)
        seriesCount = cachedSeriesInfo.Count
        categoryCount = cachedCategoryInfo.Count
    End Sub

    'parse the EIA manifest.txt file looking for the url for the bulk file associated with a category
    Public Function ReadManifest(searchCategoryId As Integer, ByRef searchDatasetUrl As String, ByRef searchDatasetModified As Date) As Boolean
        searchDatasetUrl = Nothing

        PeekToken()
        If (t <> EViewsEdx.JsonTokenType.JSON_OBJECT_BEGIN) Then Return False

        If (Not j.SkipToField("dataset")) Then Return False

        ReadToken()
        If (t <> EViewsEdx.JsonTokenType.JSON_OBJECT_BEGIN) Then Return False

        'loop over fields: each field contains a dataset objects
        While True
            'read dataset name (held in object field name)
            ReadToken()
            If (t = EViewsEdx.JsonTokenType.JSON_OBJECT_END) Then Exit While
            If (t <> EViewsEdx.JsonTokenType.JSON_FIELDNAME) Then Exit While

            Dim datasetName = val

            'check that value is an object
            ReadToken()
            If (t <> EViewsEdx.JsonTokenType.JSON_OBJECT_BEGIN) Then Return False

            'loop over fields
            Dim id As String = -1
            Dim url As String = Nothing
            Dim modified As Date
            While True
                ReadToken()
                If (t = EViewsEdx.JsonTokenType.JSON_OBJECT_END) Then Exit While
                If (t <> EViewsEdx.JsonTokenType.JSON_FIELDNAME) Then Exit While

                If (val = "category_id") Then
                    ReadToken()
                    If (t <> EViewsEdx.JsonTokenType.JSON_STRING) Then Exit While
                    id = val
                ElseIf (val = "accessURL") Then
                    ReadToken()
                    If (t <> EViewsEdx.JsonTokenType.JSON_STRING) Then Exit While
                    url = val
                ElseIf (val = "modified") Then
                    ReadToken()
                    If (t <> EViewsEdx.JsonTokenType.JSON_STRING) Then Exit While
                    modified = val  'val should contain date in ISO timestamp format
                Else
                    j.SkipValue()
                End If

            End While

            If (id = searchCategoryId) Then
                'found the desired dataset: save the URL
                searchDatasetUrl = url
                searchDatasetModified = modified
            End If

        End While

        Return Len(searchDatasetUrl) > 0

    End Function

End Class

