﻿'Database class for the EIA online database
'
'Handles the fetching of series and category information from the EIA web server
'

'for COMException
Imports System.Runtime.InteropServices

'for Web support
Imports System
Imports System.Net
Imports System.IO

'server database
Public Class ServerDatabase
    Implements EViewsEdx.IDatabase
    Implements CategoryBrowserControl.ICategorySource

    Private manager As ServerDatabaseManager

    Private tokenizer As EViewsEdx.JsonReader
    Private parser As EiaParser

    Private serverUrl As String

    'used to remember what category the browser was viewing when it was last open
    Private browserStartupCategory As Integer = -1

    'enumerator for returning search results
    Private searchEnumerator As IDictionaryEnumerator = Nothing

    'called by the database manager class when a new database is opened
    Public Sub New(ByRef mgr As ServerDatabaseManager, ByVal defaultDb As String, server As String, user As String, password As String)
        MyBase.New()

        manager = mgr
        serverUrl = server

        'create JSON tokenizer
        tokenizer = New EViewsEdx.JsonReader

        'create EIA parser
        parser = New EiaParser(tokenizer)


    End Sub

    'called by EViews immediately before the database object is deleted
    Public Sub Close() Implements EViewsEdx.IDatabase.Close
        'discard EIA parser
        parser = Nothing

        'free JSON tokenizer
        tokenizer.Detach()
        tokenizer = Nothing

        'encourage .NET to free up memory 
        GC.Collect()
    End Sub

    'return attributes for this database
    Public Function GetAttributes() As Object Implements EViewsEdx.IDatabase.GetAttributes
        Return Nothing
    End Function

    'called by EViews when the user fetches or copies a series from the database
    Public Sub ReadObject(objectId As String, destFreqInfo As String, _
                          ByRef attr As Object, ByRef vals As Object, ByRef ids As Object) Implements EViewsEdx.IDatabase.ReadObject

        'build up URL for (single) series request
        Dim apiKey As String = manager.GetApiKey()
        Dim url As String = "http://" & serverUrl & "/series/?series_id=" & StrConv(objectId, vbUpperCase) & "&api_key=" & apiKey

        Dim content As Byte() = Nothing
        If (FetchUrl(url, content)) Then

            tokenizer.AttachToData(content)

            If (parser.ReadWebReply()) Then
                If (parser.lastObjectType = "series") Then
                    parser.FetchSeriesFromCache(attr, vals, ids)
                ElseIf (parser.lastObjectType = "error" And parser.lastErrorMessage.Contains("invalid series_id")) Then
                    Throw New COMException("", EViewsEdx.ErrorCode.RECORD_NAME_INVALID)
                ElseIf (parser.lastObjectType = "error") Then
                    Throw New COMException(parser.lastErrorMessage, EViewsEdx.ErrorCode.FOREIGN_LIBRARY_ERROR)
                Else
                    'parsing error or unexpected content
                    Throw New COMException("unexpected reply from server", EViewsEdx.ErrorCode.FOREIGN_LIBRARY_ERROR)
                End If

            End If
        Else
            'fetch of url failed
            Throw New COMException("unable to fetch series from server", EViewsEdx.ErrorCode.FOREIGN_LIBRARY_ERROR)
        End If

    End Sub

    Public Sub ReadObjectAttributes(objectId As String, destFreqInfo As String, ByRef attr As Object) Implements EViewsEdx.IDatabase.ReadObjectAttributes
        'build up URL for (single) series request - uses num=0 to indcate that no data points should be fetched
        Dim url As String = "http://" & serverUrl & "/series/?series_id=" & StrConv(objectId, vbUpperCase) & "&api_key=" & manager.GetApiKey() & "&num=0"

        Dim content As Byte() = Nothing
        If (FetchUrl(url, content)) Then

            tokenizer.AttachToData(content)

            If (parser.ReadWebReply()) Then
                If (parser.lastObjectType = "series") Then
                    Dim vals As Object = Nothing
                    Dim ids As Object = Nothing
                    parser.FetchSeriesFromCache(attr, vals, ids)
                ElseIf (parser.lastObjectType = "error" And parser.lastErrorMessage.Contains("invalid series_id")) Then
                    Throw New COMException("", EViewsEdx.ErrorCode.RECORD_NAME_INVALID)
                ElseIf (parser.lastObjectType = "error") Then
                    Throw New COMException(parser.lastErrorMessage, EViewsEdx.ErrorCode.FOREIGN_LIBRARY_ERROR)
                Else
                    'parsing error or unexpected content
                    Throw New COMException("unexpected reply from server", EViewsEdx.ErrorCode.FOREIGN_LIBRARY_ERROR)
                End If

            End If
        Else
            'fetch of url failed
            Throw New COMException("unable to fetch series information from server", EViewsEdx.ErrorCode.FOREIGN_LIBRARY_ERROR)
        End If
    End Sub

    Public Sub ReadObjects(objectIds As Object, destFreqInfo As Object, ByRef attr As Object, ByRef vals As Object, ByRef ids As Object) Implements EViewsEdx.IDatabase.ReadObjects

    End Sub

    'initialize a search
    Public Sub SearchByAttributes(searchExpression As String, attrNames As String) Implements EViewsEdx.IDatabase.SearchByAttributes
        searchEnumerator = parser.GetCachedSeriesEnumerator()

    End Sub

    'return one item of search results
    Public Function SearchNext(ByRef objectId As String, ByRef attr As Object) As Boolean Implements EViewsEdx.IDatabase.SearchNext
        If (searchEnumerator.MoveNext()) Then
            objectId = searchEnumerator.Key
            Dim seriesInfo As EiaSeriesInfo = searchEnumerator.Value

            'create attribute array for EViews
            attr = New Object(4, 1) {}  '5 by 2 array
            attr(0, 0) = "description"
            attr(0, 1) = seriesInfo.description
            attr(1, 0) = "freq"
            attr(1, 1) = seriesInfo.freq
            attr(2, 0) = "start"
            attr(2, 1) = seriesInfo.startText
            attr(3, 0) = "end"
            attr(3, 1) = seriesInfo.endText
            attr(4, 0) = "last_update"
            attr(4, 1) = seriesInfo.lastUpdate

            Return True
        Else
            searchEnumerator = Nothing
            Return False
        End If

    End Function

    'called by EViews when user aborts a search (hits the Escape key)
    Public Sub SearchAbort() Implements EViewsEdx.IDatabase.SearchAbort
        searchEnumerator = Nothing
    End Sub

    'called by EViews when user clicks on the 'Browse' toolbar button.
    'returns a custom browser control allowing the user to search through this database using the category tree.
    Public Function SearchByBrowser(browserArgs As Object, ByRef attrNames As String) As Object Implements EViewsEdx.IDatabase.SearchByBrowser
        attrNames = "name,description,freq,last_update"
        Return New CategoryBrowserControl(Me, browserStartupCategory, True)
    End Function

    'callback used by the browser control to return selected results to EViews
    Public Sub PrepareBrowserSelection(selectedIds() As String) Implements CategoryBrowserControl.ICategorySource.PrepareBrowserSelection
        searchEnumerator = parser.GetSelectedSeriesEnumerator(selectedIds)
    End Sub

    Public Sub SetAttributes(attr As Object) Implements EViewsEdx.IDatabase.SetAttributes

    End Sub

    Public Sub ListObjectAttributes(ByRef attributeList As String, delim As String, ByRef scanForCustom As Boolean) Implements EViewsEdx.IDatabase.ListObjectAttributes

    End Sub

    Public Sub WriteObject(ByRef objectId As String, attr As Object, vals As Object, ids As Object, overwriteMode As EViewsEdx.WriteType) Implements EViewsEdx.IDatabase.WriteObject

    End Sub

    Public Sub WriteObjects(ByRef errors As Object, ByRef objectIds As Object, attr As Object, vals As Object, ids As Object, overwriteMode As EViewsEdx.WriteType) Implements EViewsEdx.IDatabase.WriteObjects

    End Sub

    Public Sub BeginWrite(label As String) Implements EViewsEdx.IDatabase.BeginWrite

    End Sub

    Public Sub EndWrite(reserved As Integer) Implements EViewsEdx.IDatabase.EndWrite

    End Sub

    Public Sub CopyObject(srcObjectId As String, ByRef destObjectId As String, Optional overwrite As Boolean = False) Implements EViewsEdx.IDatabase.CopyObject

    End Sub

    Public Sub DeleteObject(objectId As String) Implements EViewsEdx.IDatabase.DeleteObject

    End Sub

    Public Sub RenameObject(srcObjectId As String, destObjectId As String) Implements EViewsEdx.IDatabase.RenameObject

    End Sub

    Public Function GetCommandIds() As Object Implements EViewsEdx.IDatabase.GetCommandIds
        Return Nothing
    End Function

    Public Function DoCommand(commandId As String, args As Object) As Object Implements EViewsEdx.IDatabase.DoCommand
        Return Nothing
    End Function

    'callback used by the browser to save the current category as the startup category when browser next opens
    Public Sub SaveStartupCategory(categoryId As Integer) Implements CategoryBrowserControl.ICategorySource.SaveStartupCategory
        browserStartupCategory = categoryId
    End Sub

    'callback used by the browser to retrieve parent/child information for the specified category
    '
    'browser uses categoryId < 0 to request info on the root category (true category number will be returned by this function)
    '
    Public Function GetCategoryInfo(categoryId As Integer, ByRef parentId As Integer, ByRef description As String, ByRef categoryPath As String, _
                                    ByRef childIds() As Integer, ByRef childDescriptions() As String, _
                                    ByRef childSeriesNames() As String, ByRef childSeriesDescriptions() As String) As Integer _
                        Implements CategoryBrowserControl.ICategorySource.GetCategoryInfo

        'check whether we already have category info in our cache
        If (parser.GetCategoryState(categoryId) <= 0) Then
            'build up URL for category request
            Dim url As String
            If (categoryId >= 0) Then
                url = "http://" & serverUrl & "/category/?category_id=" & categoryId & "&api_key=" & manager.GetApiKey()
            Else
                'request root category by omitting category_id
                url = "http://" & serverUrl & "/category/?api_key=" & manager.GetApiKey()
            End If

            Dim content As Byte() = Nothing
            If (FetchUrl(url, content)) Then
                'successfully retrieved url from server
                tokenizer.AttachToData(content)

                'parse content
                If (parser.ReadWebReply()) Then
                    If (parser.lastObjectType = "category") Then
                        'if root category requested, change value to true categoryId
                        If (categoryId < 0) Then categoryId = parser.lastCategoryId
                    ElseIf (parser.lastObjectType = "error") Then
                        'EIA server returned an explicit error
                        Throw New COMException(parser.lastErrorMessage, EViewsEdx.ErrorCode.FOREIGN_LIBRARY_ERROR)
                    Else
                        'unexpected content - shouldn't happen
                        Throw New COMException("unexpected reply from server", EViewsEdx.ErrorCode.FOREIGN_LIBRARY_ERROR)
                    End If
                Else
                    'parsing error or unexpected content - shouldn't happen
                    Throw New COMException("unexpected reply from server", EViewsEdx.ErrorCode.FOREIGN_LIBRARY_ERROR)
                End If

            Else
                'fetch of url failed (http status code >= 400)
                Throw New COMException("unable to fetch category information from server", EViewsEdx.ErrorCode.FOREIGN_LIBRARY_ERROR)
            End If
        End If

        'retrieve info from cache
        parser.GetCategoryInfo(categoryId, parentId, description, categoryPath, _
                               childIds, childDescriptions, childSeriesNames, childSeriesDescriptions)
        Return categoryId
    End Function

    'process a request from the custom browser control to download the content of a category.
    '
    'currently only supports fetching top level EIA datasets by downloading an EIA bulk file.
    '
    Public Sub DownloadCategoryContent(id As Integer) Implements CategoryBrowserControl.ICategorySource.DownloadCategoryContent
        'try downloading manifest file containing urls for bulk files
        Dim manifestUrl As String = "http://" & serverUrl & "/bulk/manifest.txt"

        'note that manifest is not tagged as JSON content, so we override the content test in FetchUrl
        Dim content As Byte() = Nothing
        If (FetchUrl(manifestUrl, content, False)) Then
            'parse manifest file
            tokenizer.AttachToData(content)

            Dim bulkFileUrlText As String = Nothing
            Dim bulkFileModifiedDate As Date
            If (parser.ReadManifest(id, bulkFileUrlText, bulkFileModifiedDate)) Then
                'extract filename from end of bulk file url
                Dim lastSlashOffset As Integer = bulkFileUrlText.LastIndexOf("/"c)
                Dim bulkFileName As String = Nothing
                If (lastSlashOffset >= 0) Then
                    bulkFileName = Mid(bulkFileUrlText, lastSlashOffset + 2)
                End If

                'add API key (which sometimes seems to matter?)
                Dim bulkFileUrlWithKey = bulkFileUrlText & "?&api_key=" & manager.GetApiKey()

                Dim dialog As New DownloadDialog(bulkFileUrlWithKey, bulkFileUrlText, bulkFileModifiedDate)
                dialog.ShowDialog()

                'download with WebClient class
                'Dim request As WebClient = New WebClient
                'Dim uri As Uri = New Uri(bulkFileUrlText)
                'Dim bulkFileDirectory as String = "c:\test\"
                'request.DownloadFileAsync(uri, bulkFileDirectory & bulkFileName)

            End If
        End If
    End Sub

    'download the contents of the specified url, returning it as a byte array
    '
    'note that the following implementation assumes that the server returns 'expected' errors
    '(such as 'series name not found') with a http success status code (<400). if the server
    'returns failure status codes (>=400) an exception will be thrown by the WebRequest class
    'and any content returned will not be fetched. 
    '
    Private Shared Function FetchUrl(url As String, ByRef content As Byte(), Optional requireJsonContentType As Boolean = True) As Boolean
        Try
            Dim wrGETURL As HttpWebRequest
            wrGETURL = WebRequest.Create(url)

            Dim objStream As Stream
            objStream = wrGETURL.GetResponse.GetResponseStream()

            Dim headers As WebHeaderCollection = wrGETURL.GetResponse.Headers
            If (Left(wrGETURL.GetResponse.ContentType, 16) = "application/json" Or Not requireJsonContentType) Then
                Dim byteCount As Long = wrGETURL.GetResponse.ContentLength

                If (byteCount >= 0) Then
                    'content length was provided by server - allocate buffer
                    content = New Byte(byteCount - 1) {}

                    'fetch content into buffer
                    Dim totalBytesRead As Integer = 0
                    While (totalBytesRead < byteCount)
                        Dim bytesRead As Integer = objStream.Read(content, totalBytesRead, byteCount - totalBytesRead)
                        If (bytesRead = 0) Then Return False 'shouldn't happen if content_length was explicitly specified
                        totalBytesRead = totalBytesRead + bytesRead
                    End While

                    Return True
                Else
                    'no content length provided by server- need to resize output buffer dynamically

                    'allocate initial buffer
                    byteCount = 4000
                    content = New Byte(byteCount - 1) {}

                    'fetch content into buffer
                    Dim totalBytesRead As Integer = 0
                    While (totalBytesRead < byteCount)
                        Dim bytesRead As Integer = objStream.Read(content, totalBytesRead, byteCount - totalBytesRead)
                        If (bytesRead = 0) Then Exit While
                        totalBytesRead = totalBytesRead + bytesRead
                        If (totalBytesRead = byteCount) Then
                            ReDim Preserve content(byteCount * 2 - 1)
                            byteCount = byteCount * 2
                        End If
                    End While
                    ReDim Preserve content(totalBytesRead - 1)

                    Return True
                End If
            End If

        Catch e As System.Net.WebException
            'server probably returned an http status code indicating an error (>=400)
        End Try

        'failed
        content = Nothing
        Return False

    End Function

End Class


